program symos_chmu
! SYMOS ČHMÚ. Program pro výpočet rozptylu znečišťujících látek podle 
! Metodické příručky k modelu SYMOS’97 - aktualizace 2013 (Příloha 1 
! Metodického pokynu odboru ochrany ovzduší MŽP ke zpracování rozptylových 
! studií viz <http://www.mzp.cz/cz/zpracovani_rozptylovych_studii_metodika>. 
! Odlišnosti od Metodické příručky jsou uvedeny v Uživatelské dokumentaci
! modelu.
!
! Copyright (C) 2016 Český hydrometeorologický ústav; 
! Kontaktní osoby (pouze pro zasílání připomínek; ČHMÚ neposkytuje technickou
! podporu): 
!     Nina Benešová <nina.benesova@chmi.cz>, 
!     Ondřej Vlček  <ondrej.vlcek@chmi.cz>
!
! Tento program je svobodným softwarem: můžete jej používat, šířit a 
! upravovat podle ustanovení Obecné veřejné licence GNU (GNU General Public 
! Licence), vydávané Free Software Foundation, a to na základě 3. verze 
! této Licence, nebo (podle vašeho uvážení) kterékoli pozdější verze.
!
! Tento program je rozšiřován v naději, že bude užitečný. Ačkoliv před 
! uvolněním pro veřejnost prochází pečlivým testováním, NEMUSÍ BÝT ZARUČENÁ 
! JEHO PLNÁ FUNKČNOST A STAŽENÍM ZA NĚJ NEJSOU POSKYTOVÁNY JAKÉKOLIV ZÁRUKY 
! VČ. ZÁRUKY ZA VADY a to i tehdy, pokud rozhodným právem nebude právo 
! České republiky.  Při spuštění jakékoliv jeho verze vč. aktualizací není 
! možné požadovat jakékoliv náhrady škody, pokud vznikají jeho užíváním, 
! šířením a úpravou či nesprávným instalováním na jakémkoliv zařízení. 
! Nejsou poskytovány ani odvozené záruky PRODEJNOSTI anebo VHODNOSTI PRO 
! URČITÝ ÚČEL. 
!
! Další podrobnosti hledejte v Obecné veřejné licenci GNU, jejíž kopii jste 
! měli obdržet spolu s tímto programem. Pokud se tak nestalo, najdete ji zde:
! http://www.gnu.org/licenses/gpl.html.
!
! Veškeré práce využívající tento program musí obsahovat následující citaci: 
! „ČHMÚ. SYMOS ČHMÚ v<číslo verze> [software]. [přístup <datum>]. 
!  Dostupné z: <odkaz na www>. K dispozici pro Windows a Linux.“ 
! Upravené verze programu musí být náležitě označeny a obsahovat popis
! provedených změn.

  use mod_init
  use mod_funkce
  !$ use omp_lib                                        ! modul s rozhranim funkci a podprogramu OpenMP

  implicit none

  character(10)				  :: verze = 'v1.1.2'

  integer  :: pocet_ref_bodu, pocet_zdroju, pocet_output_bodu	! pocty prislusnych bodu
  integer  :: si, ri, ti, n										! pomocne indexy v cyklech
  integer  :: met, AllocateStatus, chunk_size
  type(zdroj)				  :: izdroj
  type(ref_bod)				  :: ibod
  integer(8)  				  :: clck_counts_beg, clck_counts_end, clck_rate
  real(dp), dimension(:,:,:), allocatable :: c          ! koncentrace pro dany ref. bod pro vsechny podminky (v poradi smery vetru, rychlosti, tridy stability)
  real(dp), dimension(:,:,:), allocatable :: c_out      ! vystup z subroutiny evaluate_conc
  real(dp), dimension(:,:,:), allocatable :: c_d        ! denni maxima
  real(dp), dimension(:,:,:), allocatable :: cd_pom     ! pomocne pole
  real(dp)				  :: c_ave						! prispevek dosud spoctenych zdroju k aktualnimu ref. bodu k rocnimu prumeru
  real(dp), dimension(:,:), allocatable   :: c_max      ! pro ucely vypoctu maximalnich hodinovych koncentraci
  real(dp), dimension(:,:), allocatable   :: cd_max     ! pro ucely vypoctu maximalnich dennich koncentraci
  real(dp), dimension(:,:,:,:), allocatable :: th_exceed, td_exceed  ! pro ucely vypoctu dob prekroceni
  real(dp), dimension(:), allocatable     :: c_t        ! koncentrace v referencnim bode pres vsechny casy 
  real(dp), dimension(:), allocatable     :: c_r        ! pole pro vypocet prumeru z hodinoveho vypoctu
  real(dp), dimension(:), allocatable     :: c_m        ! pole pro vypocet maxima z hodinoveho vypoctu
  real(dp), dimension(:,:), allocatable   :: results_t  ! pole pro uchovani casovych rad pro vypis
  real(dp), dimension(:,:), allocatable   :: results	! vysledne statistiky pro vsechny referencni body
  character(250) :: user_File							! nazev souboru uzivatelem definovanymi parametry
  character(3)	 :: num_threads							! pocet vlaken pro paralelni vypocet
  character*1    :: creturn = achar(13)					! vraci ASCII polohu znaku carriage return
  logical	 :: is_metodika								! indikuje zda probiha vypocet dle metodiky

  write(*,*) 
  write(*,*) 'SYMOS CHMU ',trim(verze), ' Copyright (C) 2016 Cesky hydrometeorologicky ustav'
  write(*,*) 'Tento program je ABSOLUTNE BEZ ZARUKY. Podrobnosti najdete v licencnich podminkach ', &
  	     'uvedenych v souboru src/LICENSE.pdf'
  write(*,*) 

  ! nacti vstupni data a parametry vypoctu zadane uzivatelem
  call read_cmd_arguments(user_File, num_threads)
  !$ call set_num_threads(num_threads)
  call read_input_data(user_File)

  ! nacti parametry definovane metodikou
  call read_param 

  ! budeme pocitat dle metodiky?  
  is_metodika = is_max_abs .or. is_average .or. is_max_11 .or. is_exceed

  ! -> nasledujici dve promenne odsunout nekam do modulu
  ! pocet ref. bodu, zdroju a cas. okamziku
  pocet_ref_bodu = size(ref_body)
  pocet_zdroju = size(zdroje)

  ! priprava pro hodinovy vypocet ------------------------------------------------------------------------------
  if (is_time) then
    ntime = size(w_dir, 2)   
    pocet_output_bodu = count(ref_body%output)
    allocate(results_t(ntime, pocet_output_bodu), STAT = AllocateStatus)
    if (AllocateStatus /= 0) stop "*** Not enough memory to allocate results_t ***"
    results_t = 0
    allocate(c_r(pocet_ref_bodu))
    allocate(c_m(pocet_ref_bodu))
    c_r = 0; c_m = 0

    ! pokud mame na vstupu teplotni gradient tak ho prepocti na tridy stability
    if (is_tep_gr) then
        do met = 1, met_num
           call get_tridy_st(tep_gr(met, :), tridy_st(met, :))
           if (inverze) call get_tridy_st(tep_gr_i(met, :), tridy_st_i(met, :))
        enddo
    end if
  endif

  ! priprava pro vypocet podle metodiky ----------------------------------------------------------------
  if (is_metodika) then

    ! alokace pole pro vysledky
    allocate(results(size(stats), pocet_ref_bodu))
    results = 0

    ! vypocet rychlostnich intervalu
    call u_range_gen(is_max_abs, u_range, nu)
  endif


  ! hlavni cyklus pro vypocet koncentraci --------------------------------------------------------------
  if (is_metodika .or. is_time) then
	  call system_clock( clck_counts_beg, clck_rate )
	  write(*,*) 'Vypocet koncentraci'  

	  n = 0
	  !$OMP PARALLEL PRIVATE(ibod, izdroj, c_t, c, c_out, c_d, cd_pom, c_max, cd_max, c_ave, th_exceed, td_exceed) 
	  if (is_time) allocate(c_t(ntime))
	  if (is_metodika) then
		 allocate(c_out(nphi, maxval(nu), n_stab_cl))
		 allocate(c(nphi, maxval(nu), n_stab_cl))
		 allocate(c_d(nphi, maxval(nu), n_stab_cl))
		 allocate(cd_pom(nphi, maxval(nu), n_stab_cl))
		 allocate(c_max(nphi, poc_rozpt_t))
		 allocate(cd_max(nphi, poc_rozpt_t))
		 allocate(th_exceed(nphi, poc_rozpt_t, num_lim, 2))
		 allocate(td_exceed(nphi, poc_rozpt_t, num_lim, 2))  
	  endif

	  ! magicka formule pro vypocet velikosti bloku pro jedno vlakno
	  !$ chunk_size = pocet_ref_bodu / omp_get_num_threads() / 4
	  !$ if (chunk_size .eq. 0) chunk_size = 1


	  !$OMP DO SCHEDULE(dynamic, chunk_size) 
	  do ri = 1, pocet_ref_bodu              ! cyklus pres referencni body     
		  ibod = ref_body(ri)                ! ref. bod ktery pocitam pri aktualnim pruchodu

		  if (is_time) c_t = 0
		  if (is_metodika) then
		     c = 0 
		     c_d = 0
		     c_max = 0
		     cd_max = 0
		     th_exceed = 0
		     td_exceed = 0
		     c_ave = 0
		  endif
		  
		  do si = 1, pocet_zdroju            ! cyklus pres zdroje 
		      izdroj = zdroje(si)            ! zdroj ktery pocitam pri aktualnim pruchodu
		      
		      ! vypocet theta, delta, x, zmax
			  x = eval_x(ibod%x, ibod%y, izdroj%x, izdroj%y)
		      if ((x < infl_dist_min) .or. (x > infl_dist_max)) cycle
		      call eval_theta_delta(ibod%x, ibod%y, ibod%z, izdroj%x, izdroj%y, izdroj%z, theta, delta, zmax, x)

		      ! cyklus pres casove kroky
		      if (is_time) then
		         do ti = 1, ntime	   	
		            c_t(ti) = c_t(ti) + time_eval_conc(ti, ri, izdroj)
		         end do   
		      endif
		 
		      ! vypocet dle metodiky
		      if (is_metodika) then
		         call evaluate_conc(ri, izdroj, c_out)
		         c = c + c_out   
		         if (is_daily) then
		             cd_pom = daily(c_out, izdroj%Pd, daily_type, nphi, maxval(nu), n_stab_cl)
		             c_d = c_d + cd_pom
		         endif        
		         c_max = c_max + c_class(c_out)   
		         cd_max = cd_max + c_class(cd_pom) 
		         if (is_exceed) then    
		            call prepare_exceed(c_max, th_exceed, ch_exceed, izdroj%met, izdroj%alfa, num_lim, poc_rozpt_t, nphi)
		            if (is_daily) call prepare_exceed(cd_max, td_exceed, cd_exceed, izdroj%met, izdroj%alfa, num_lim, poc_rozpt_t, nphi)
		         endif
		         if (is_average) c_ave = c_ave + eval_average(c_class(c_out), ruzice(:, :, izdroj%met), izdroj%alfa)
		      endif
		      
		  end do

		  ! ulozeni potrebnych dat pro hodinovy vypocet
		  if (is_time) then
		     if (ibod%output) then
		        results_t(:, locate_string(pack(ref_body%id, ref_body%output), ibod%id)) = c_t
		     endif
		     c_r(ri) = c_r(ri) + sum(c_t)
		     c_m(ri) = maxval(c_t)
		  endif

		  ! vyhodnoceni vsech statistik pro dany referencni bod pro metodiku
		  if (is_metodika) call eval_stat(c, c_d, c_max, cd_max, c_ave, th_exceed, td_exceed, results(:, ri))

		  !$OMP CRITICAL
		  n = n + 1
		  !$OMP END CRITICAL
		  write(*, 101, advance = 'no' ) creturn, n, pocet_ref_bodu   
		end do    
        !$OMP END DO

		if (allocated(c_t)) deallocate(c_t)
		if (allocated(c)) deallocate(c)
		if (allocated(c_out)) deallocate(c_out)
		if (allocated(c_d)) deallocate(c_d)
		if (allocated(cd_pom)) deallocate(cd_pom)
		if (allocated(c_max)) deallocate(c_max)
		if (allocated(cd_max)) deallocate(cd_max)
		if (allocated(th_exceed)) deallocate(th_exceed)
		if (allocated(td_exceed)) deallocate(td_exceed)
		!$OMP END PARALLEL

		write(*,*)
		call system_clock( clck_counts_end, clck_rate)
		write(*, 100) real(clck_counts_end - clck_counts_beg)/real(clck_rate)

	 
	  ! konec hlavni cyklus pro vypocet koncentraci --------------------------------------------------------------

	  ! vypis vysledku -------------------------------------------------------------------------------------------
		write(*,*) 'Vypis vysledku'

		! vypis hodinovych vysledku  
		if (is_time) then  
		   call write_output_time(oputDir, output_time, pack(ref_body%id, ref_body%output), date, results_t)
		   c_r = c_r / ntime
		   call write_time_average(oputDir, output_time, c_r)
		   call write_time_maximum(oputDir, output_time, c_m)
		endif

		! vypis vysledku dle metodiky
		if (is_max_abs .or. is_average .or. is_max_11 .or. is_exceed) call write_output_stat(results)

	  100 format(F15.2, ' s')
	  101 format(a, '   - spocten ref. bod c. ', i9, '/', i9)
  else
      write(*,*) 
      write(*,*) "Nebyl zapnut zadny vypocet. Zkontrolujte prislusne promenne v konfiguracnim souboru."
      write(*,*) "Vypoct casove rady se zapina pomoci is_time, vypocet dle metodiky pak promennymi is_average, ", &
                 "is_max_abs, is_max_11, is_daily nebo is_exceed = .TRUE." 
      write(*,*) "Konec programu."
  endif


  ! dealokace promennych, neni potreba
  ! zatim tu je jako seznam promennych, ktere zustaly alokovane
  if (allocated(ref_body)) deallocate(ref_body)
  if (allocated(zdroje)) deallocate(zdroje)
  if (allocated(z)) deallocate(z)

  if (allocated(c_r)) deallocate(c_r)
  if (allocated(results_t)) deallocate(results_t)
  if (allocated(w_dir)) deallocate(w_dir) 
  if (allocated(w_sp)) deallocate(w_sp)
  if (allocated(tep_gr)) deallocate(tep_gr)
  if (allocated(tridy_st)) deallocate(tridy_st)
  if (allocated(w_dir_i)) deallocate(w_dir_i) 
  if (allocated(w_sp_i)) deallocate(w_sp_i)
  if (allocated(tep_gr_i)) deallocate(tep_gr_i)
  if (allocated(tridy_st_i)) deallocate(tridy_st_i)


contains

 ! -------------------------------------------------------------------------------------------------------------------------------
 !> Načte argumenty z příkazové řádky (souboru s uživatelem definovanými proměnnými a počet vláken pro paralelní běh programu) 

 subroutine read_cmd_arguments(user_file, num_threads)
    implicit none
    character(*), intent(  out) :: user_file    !< název souboru s uživatelem definovanými proměnnými
    character(*), intent(  out) :: num_threads  !< počet vláken pro paralelní běh programu
    integer                     :: num_arg

    num_arg = iargc() 
    if ((num_arg == 0) .or. (num_arg > 2)) then 
        write(*,*) 'Pouziti programu:'// new_line('A') // &
                   'symos.exe <soubor se vstupnimi parametry> [<pocet vlaken>]'
        stop
    endif
    call get_command_argument(1, user_File)      
    call get_command_argument(2, num_threads)  

 end subroutine read_cmd_arguments


 ! -------------------------------------------------------------------------------------------------------------------------------
 !> Nastaví počet vláken dle uživatelem definované hodnoty num_threads. Nic znamená maximum.

 subroutine set_num_threads(num_threads)
    implicit none
    character(*), intent(in   ) :: num_threads  !< počet vláken pro paralelní běh programu
    integer                     :: numeric, ios

    if (num_threads == '') then
       !$ call omp_set_num_threads(omp_get_max_threads())
       return
    else
       read(num_threads, *, iostat = ios) numeric
       if ((numeric < 1) .or. (ios /= 0)) then
          write(*,*) 'Neplatna hodnota poctu vlaken.'
          stop
       endif
       !$ call omp_set_num_threads(numeric)
    endif

 end subroutine set_num_threads

end program symos_chmu












