module mod_funkce
! SYMOS ČHMÚ. Program pro výpočet rozptylu znečišťujících látek podle 
! Metodické příručky k modelu SYMOS’97 - aktualizace 2013 (Příloha 1 
! Metodického pokynu odboru ochrany ovzduší MŽP ke zpracování rozptylových 
! studií viz <http://www.mzp.cz/cz/zpracovani_rozptylovych_studii_metodika>. 
! Odlišnosti od Metodické příručky jsou uvedeny v Uživatelské dokumentaci
! modelu.
!
! Copyright (C) 2016 Český hydrometeorologický ústav; 
! Kontaktní osoby (pouze pro zasílání připomínek; ČHMÚ neposkytuje technickou
! podporu): 
!     Nina Benešová <nina.benesova@chmi.cz>, 
!     Ondřej Vlček  <ondrej.vlcek@chmi.cz>
!
! Tento program je svobodným softwarem: můžete jej používat, šířit a 
! upravovat podle ustanovení Obecné veřejné licence GNU (GNU General Public 
! Licence), vydávané Free Software Foundation, a to na základě 3. verze 
! této Licence, nebo (podle vašeho uvážení) kterékoli pozdější verze.
!
! Tento program je rozšiřován v naději, že bude užitečný. Ačkoliv před 
! uvolněním pro veřejnost prochází pečlivým testováním, NEMUSÍ BÝT ZARUČENÁ 
! JEHO PLNÁ FUNKČNOST A STAŽENÍM ZA NĚJ NEJSOU POSKYTOVÁNY JAKÉKOLIV ZÁRUKY 
! VČ. ZÁRUKY ZA VADY a to i tehdy, pokud rozhodným právem nebude právo 
! České republiky.  Při spuštění jakékoliv jeho verze vč. aktualizací není 
! možné požadovat jakékoliv náhrady škody, pokud vznikají jeho užíváním, 
! šířením a úpravou či nesprávným instalováním na jakémkoliv zařízení. 
! Nejsou poskytovány ani odvozené záruky PRODEJNOSTI anebo VHODNOSTI PRO 
! URČITÝ ÚČEL. 
!
! Další podrobnosti hledejte v Obecné veřejné licenci GNU, jejíž kopii jste 
! měli obdržet spolu s tímto programem. Pokud se tak nestalo, najdete ji zde:
! http://www.gnu.org/licenses/gpl.html.
!
! Veškeré práce využívající tento program musí obsahovat následující citaci: 
! „ČHMÚ. SYMOS ČHMÚ v<číslo verze> [software]. [přístup <datum>]. 
!  Dostupné z: <odkaz na www>. K dispozici pro Windows a Linux.“ 
! Upravené verze programu musí být náležitě označeny a obsahovat popis
! provedených změn.


 implicit none

 integer, parameter		:: dp = kind(0.d0)   	        ! double precision               
 integer			:: ntime 		        ! pocet nactenych casovych kroku
 real(dp), parameter 		:: Pi  = acos(-1._dp)
 real(dp), parameter 		:: T0K   = 273.15_dp  		! 0 °C v Kevinech
 real(dp), parameter 		:: pnorm = 101325._dp 		! normalní tlak
 real(dp), parameter		:: g = 9.81_dp			! [m/s^2] tihove zrychleni
 real(dp), parameter		:: gamma_d = 0.01_dp            ! [°C/m] adiabaticky vertikalni teplotni gradient
 real(dp), parameter		:: Lv = 2.5d6    		! [J/kg] latentni teplot vyparu vody	
 real(dp), parameter		:: cp = 1004._dp		! [J/kg K] merne teplo pri konstantnim tlaku
 real(dp), parameter		:: Er = 611._dp			! [Pa] tlak nasycenych par pri teplote 0 °C
 real(dp), parameter		:: Rv = 461._dp                 ! [J/kg K] plynova konstanta pro vodni paru
 real(dp), parameter		:: Rd = 287._dp                 ! [J/kg K] plynova konstanta pro suchy vzduch

 character(50), parameter	:: par_File = 'param.nml'       ! nml soubor se vstupnimi parametry vypoctu
 integer, parameter     	:: n_stab_cl = 5  ! nemenit     ! pocet trid stability, kod neni pripraven na zmenu tohoto parametru
 integer, parameter 		:: max_class_w_sp = 3 ! nemenit ! pocet tridnich rychlosti, kod neni pripraven na zmenu tohoto parametru
 integer, parameter 		:: max_k_u = 3			! pocet trid pro koef. odstranovani k_u
 integer, parameter 		:: num_Fz_levels = 26           ! pocet hodnot pro funkci F(z), tab 3.8
 real(dp)			:: uhel_limit_b, uhel_limit_pl

 real(dp), dimension(max_class_w_sp)     :: class_w_sp          ! tridni rychlosti
 real(dp), dimension(max_class_w_sp - 1) :: class_w_sp_UE       ! horni hranice pro zarazeni rychlosti do dane tridy   

 ! konstanty pro výpočet rozptylových parametrů (Metodika SYMOS, kap. 3.2.5.1)
 real(dp), dimension(n_stab_cl) :: ay_1h, by_1h, az_1h, bz_1h, &     ! pro   hodinové hodnoty imisních koncentrací 
                                   ay_8h, by_8h, az_8h, bz_8h, &     ! pro  8hodinové hodnoty imisních koncentrací
                                   ay_30m, by_30m, az_30m, bz_30m, & ! pro 30minutové hodnoty imisních koncentrací
                                   ay, by, az, bz                    ! s jiz prirazenou hodnotou podle prumerovaciho intervalu

 ! konstanty Ks, Km a ε pro výpočet efektivní výšky odpovidajici jednotlivym tridam stability (Metodika SYMOS, kap. 3.2.3.1)
 real(dp), dimension(n_stab_cl)      :: Ks, Km, Eps 

 ! konstanty pro doby setrvani
 real(dp)			:: k_u_all(max_k_u), k_u, k_p(n_stab_cl)
 
 ! promenne souvicejici s terenem a vzajemnou polohou dvojice zdroj-ref.bod
 real(dp) :: theta, zmax, delta, x
 !$OMP THREADPRIVATE(delta, theta, zmax, x)

 ! rychlostni interval
 real(dp), dimension(:), allocatable :: u_range
 integer, dimension(n_stab_cl) :: nu

 ! zeslabeni vlivu nizkych zdroju na horach
 real(dp)			:: Fz(2, num_Fz_levels)
			
 ! exponent pro mocniny profil vetru v jednotlivych tridach stability (Metodika SYMOS, kap. 3.2.4.1)
 real(dp), dimension(n_stab_cl) :: u_p
 real(dp), dimension(n_stab_cl) :: stab_cl_UE 		! horni hranice (<) intervalu stabilitnich trid (u posledniho se neuvadi)

 ! promenne souvicejici s vypoctem chladicich vezi
 real(dp)			:: gamma(3)             ! vertikalni tepl. gradient

 ! seznam vysledku k ukladani a vypisu
 ! pokud zmenim nazev vystupu nebo neco pridam musim provest prislusne zmeny v subroutine eval_stat a write_output 
 ! zmena poradi nevadi (ale T jsou za sebou a 11 maxim taky)
 character(12) :: stats(37) = (/ character(len=12) :: 'ave', &
                                                      'Ch_max', 'Ch_class_max', 'Ch_w_sp_max', 'Ch_w_dir_max', &  
						      'Ch_I_1.7', 'Ch_II_1.7', 'Ch_II_5', &
                                                      'Ch_III_1.7', 'Ch_III_5', 'Ch_III_11', &
                                                      'Ch_IV_1.7', 'Ch_IV_5', 'Ch_IV_11', 'Ch_V_1.7', 'Ch_V_5', &
                                                      'Th1', 'Th2', 'Th3', &                                                     
                                                      'Cd_max', 'Cd_class_max', 'Cd_w_sp_max', 'Cd_w_dir_max', &
						      'Cd_I_1.7', 'Cd_II_1.7', 'Cd_II_5', &
                                                      'Cd_III_1.7', 'Cd_III_5', 'Cd_III_11', &
                                                      'Cd_IV_1.7', 'Cd_IV_5', 'Cd_IV_11', 'Cd_V_1.7', 'Cd_V_5', &
                                                      'Td1', 'Td2', 'Td3'/)

 namelist/param/  ay_1h, by_1h, az_1h, bz_1h, &
                  ay_8h, by_8h, az_8h, bz_8h, &
                  ay_30m, by_30m, az_30m, bz_30m, &
                  Ks, Km, Eps, u_p, &
                  stab_cl_UE, class_w_sp, class_w_sp_UE, &
                  uhel_limit_b, uhel_limit_pl, k_u_all, k_p, Fz, gamma
	
contains



! -------------------------------------------------------------------------------------------------------------------------------

subroutine read_param
 ! nacte parametry vypoctu definovane metodikou ze souboru nmlFile
 ! vypocita a nastavi nektere parametry vypoctu

 use mod_init

 implicit none

 integer			:: nmlUnit, ierror, ios
 character(400)			:: error_msg
 
 error_msg = 'parametry vypoctu'
 call check_file_exist(par_File, ierror, error_msg)
 if (ierror /= 0) call raise_error(error_msg)
 open(unit = newunit(nmlUnit), file = par_File, delim = 'apostrophe')
  read(nmlUnit, nml = param, iostat = ios)  
  if (ios /= 0) then
     call error_nml_read(par_File, ios, ierror, error_msg)
     call raise_error(error_msg)
  endif
 close(nmlUnit) 

 ! koeficient pro dobu setrvani
 k_u = get_coef_dep(trida_lat)

 ! pro vypocet sigma
 call get_sgm_par(shortterm_avg)

end subroutine read_param

! -------------------------------------------------------------------------------------------------------------------------------

subroutine evaluate_conc(ri, izdroj, c_out)
  ! pocitej koncentrace v ri-tem bode od zdroje izdroj
  ! vystup: c_out koncentrace pro vsechny azimuty, vsechny rychlosti a vsechny tridy stability

  use mod_init

  implicit none

  integer,     intent(in   ) 				:: ri 			!< vstupujici index referencniho bodu
  type(zdroj), intent(inout) 				:: izdroj 		!< zdroj ktery pocitam
  real(dp), dimension(:, :, :), intent(  out)		:: c_out                !> koncentrace v zavislosti na smeru vetru, tride stab a rychlosti
  integer						:: ci, ui, ai
  real(dp) 					        :: phi,lambda       
  real(dp)						:: h, hl, u_hl, Kh, z1, z2, z3, z4
     
  c_out = 0
  izdroj%emis_fact = 1

!  ci = 1
!  ui = 1.5
!  ai = 331

  do ci = 1, n_stab_cl ! cyklus pres tridy stability
    do ui = 1, nu(ci) ! cyklus pres rychlostni tridy
	call evaluate_par(izdroj, ci, ri, u_range(ui), 10._dp, 0._dp, h, hl, z1, z2, z3, z4, Kh, u_hl)

        do ai = 1, nphi  ! cyklus pres azimuty vetru     
            phi = (ai - 1) * phi_step         

            lambda = get_lambda(phi, h, 10._dp, delta, wind_rotation)    
!            write(*,*) 'delta1 = ', delta - (h - 10._dp) / 25
!            write(*,*) 'lambda = ', lambda
!            write(*,*) 'z1 = ', z1, 'z2 = ', z2, 'z3 = ',z3
	    call evaluate_gauss(ci, izdroj, x, theta, lambda, u_hl, Kh, z1, z2, z3, z4, &
                                hl, phi, .false., c_out(ai, ui, ci)) 
!            write(*,*) 'c = ', c_out(ai, ui, ci)

         end do ! cyklus pres azimuty vetru
      end do ! cyklus pres rychlostni tridy

  end do ! cyklus pres stabilitu    

end subroutine evaluate_conc



! -------------------------------------------------------------------------------------------------------------------------------

!> Pomocna funkce pro 'evaluate_conc' a 'time_eval_conc' vyhodnucujici parametry potrebne k vypoctu rozptylu
!> -----------

subroutine evaluate_par(izdroj, ci, ri, u_r, u_height, L, h, hl, z1, z2, z3, z4, Kh, u_hl)

  use mod_init, only: zdroj, free_atmo, ref_body
  implicit none

  type(zdroj), intent(in   ) :: izdroj	 !< zdroj
  integer,     intent(in   ) :: ci	 !< trida stability
  integer,     intent(in   ) :: ri	 !< index ref. bodu
  real(dp),    intent(in   ) :: u_r	 !< rychlost vetru ve vysce u_height
  real(dp),    intent(in   ) :: u_height !< vyska ve ktere je zadan vitr u_r
  real(dp),    intent(in   ) :: L	 !< vyska inverze, v pripade bez inverze je hodnota libovolna a vysledne z4 se nepouzije
  real(dp),    intent(  out) :: h, hl, z1, z2, z3, z4, Kh, u_hl !< vystupni promenne
  real(dp)		     :: u_10

  call hef(izdroj, ci, u_r, u_height, x, zmax, h, hl)           
  u_hl = u_h(hl, u_r, u_height, u_p(ci))                ! pocita rychlost vetru v efektivni vysce zdroje hl
  u_10 = u_h(10._dp, u_r, u_height, u_p(ci))            ! pocita rychlost vetru ve vysce 10 m 
  Kh = get_Kh(Fz, num_Fz_levels, izdroj%z, hl, ref_body(ri)%z, ci, u_10)

  call get_vert_coord(ref_body(ri)%z, izdroj%z, ref_body(ri)%l, hl, L, free_atmo, z1, z2, z3, z4)
!  write(*,*) 'h = ', h, ' hl = ', hl
!  write(*,*) 'Kh = ', Kh, ' u_hl = ', u_hl

end subroutine evaluate_par

! -------------------------------------------------------------------------------------------------------------------------------

!> Pomocna funkce pro 'evaluate_conc' a 'time_eval_conc' vyhodnucujici Gaussovsky rozptyl vlecky  
!> -----------
subroutine evaluate_gauss(ci, izdroj, x, theta, lambda, u_hl, Kh, z1, z2, z3, z4, hl, phi, inv, c)

  use mod_init

  implicit none
  
  integer,     intent(in   ) :: ci					   !< trida stability
  type(zdroj), intent(in   ) :: izdroj					   !< zdroj
  real(dp),    intent(in   ) :: x 					   !< vzdalenost zdroje a ref. bodu
  real(dp),    intent(in   ) :: lambda, u_hl, Kh, z1, z2, z3, z4, hl, phi  !< pomocne promenne
  real(dp),    intent(in   ) :: theta
  logical,     intent(in   ) :: inv
  real(dp),    intent(  out) :: c					   !< vysledna koncentrace
  real(dp) :: sgmy, sgmz, sgmy0, sgmz0, xL, yL, uhel_limit
  
  if (izdroj%typ == 1) then
      uhel_limit = uhel_limit_b
  else
      uhel_limit = uhel_limit_pl
  endif

  if ((lambda <= uhel_limit) .or. (lambda >= (360 - uhel_limit))) then 
      xL = x * cos(lambda * pi / 180) 
      yL = x * sin(lambda * pi / 180)
      call get_sgm0(izdroj, phi, az(ci), bz(ci), sgmy0, sgmz0) 
      call get_sgm(xL, ay(ci), by(ci), az(ci), bz(ci), sgmy0, sgmz0, sgmy, sgmz)
!      write(*,*) 'sgmy0 = ', sgmy0, 'sgmz0 = ', sgmz0, 'sgmy = ', sgmy-sgmy0, 'sgmz = ', sgmz-sgmz0
      c = konc_norm(xL, u_hl, izdroj%Vs, yL, k_u, Kh, z1, z2, z3, hl, theta, sgmy, sgmz)
      if (inv) c = c + inv_exp(xL, u_hl, izdroj%Vs, yL, k_u, Kh, z4, hl, sgmy, sgmz)
      c = izdroj%emis_fact * c * (izdroj%Mz + izdroj%Mz_NO * fac_NO(is_no, k_p(ci), xL, u_hl))  
  else
      c = 0
  end if 

end subroutine evaluate_gauss

! -------------------------------------------------------------------------------------------------------------------------------

function c_class(c)

  use mod_init

  implicit none
  
  integer                     :: ci
  real(dp), dimension(:,:,:), intent(in   )     :: c
  real(dp)                    :: c_class(nphi, poc_rozpt_t)  
  integer :: ui_ind1, ui_ind2, ui_ind3		! pomocne indexy pri nachazeni maximalnich koncentraci v tridach stability

  ! najde indexy, pro ktere se v nasem rychlostnim intervalu vyskytuje zadana rychlost dle metodiky
  
  do  ci = 1, n_stab_cl
  ! OSETRIT SITUACI KDY TAM INDEX NENAJDE I KDYZ BY TAM BYT MEL	
    ui_ind1 = find(u_range(:nu(ci)), class_w_sp(1)) 
    ui_ind2 = find(u_range(:nu(ci)), class_w_sp(2)) 
    ui_ind3 = find(u_range(:nu(ci)), class_w_sp(3))

    if (ci == 1) then
      c_class(:,1) = c(:, ui_ind1, 1)
	      
    else if (ci == 2) then
      c_class(:,2) = c(:, ui_ind1, 2)
      c_class(:,3) = c(:, ui_ind2, 2)
	  
    else if (ci == 3) then
      c_class(:,4) = c(:, ui_ind1, 3)
      c_class(:,5) = c(:, ui_ind2, 3)
      c_class(:,6) = c(:, ui_ind3, 3)

    else if (CI == 4) then
      c_class(:,7) = c(:, ui_ind1, 4)
      c_class(:,8) = c(:, ui_ind2, 4)
      c_class(:,9) = c(:, ui_ind3, 4)
	
    else if (CI == 5) then
      c_class(:,10) = c(:, ui_ind1, 5)
      c_class(:,11) = c(:, ui_ind2, 5) 

    endif
  
  end do

end function c_class


! -------------------------------------------------------------------------------------------------------------------------------
!> Pro casovy okamzik `ti` spocita prispevek zdroje izdroj v ref. bode `ri`
!> -------------------------
!>

real(dp) function time_eval_conc(ti, ri, izdroj)
 
  use mod_init

  implicit none
    
  integer,     intent(in   ) :: ti     !< index tridy stability
  integer,     intent(in   ) :: ri     !< index referencniho bodu
  type(zdroj), intent(inout) :: izdroj !< zdroj ktery pocitam
  integer		 :: ci, met
  real(dp) 		 :: lambda       	
  real(dp)		 :: h, hl, u_hl, Kh, z1, z2, z3, z4
  logical		 :: inverze_eval
     
  met = izdroj%met
  izdroj%emis_fact = time_emis_fact(izdroj%time_prof, ti)

  inverze_eval = .FALSE.
  if (inverze) then
      if (int(zi(met, ti)) .ne. -9999) inverze_eval = .TRUE.
  endif
      
  if (inverze_eval) then
    call eval_inv(ti, ri, izdroj, time_eval_conc)
  else
    ci = tridy_st(met, ti)
    call evaluate_par(izdroj, ci, ri, w_sp(met, ti), u_height, 0._dp, h, hl, z1, z2, z3, z4, Kh, u_hl)
    lambda = get_lambda(w_dir(met, ti), h, u_height, delta, wind_rotation)	
!             write(*,*) 'x = ', x
!             write(*,*) 'delta = ', delta - (h - u_height) / 25
!             write(*,*) 'lambda = ', lambda
!             write(*,*) 'z1 = ', z1, 'z2 = ', z2, 'z3 = ',z3
    call evaluate_gauss(ci, izdroj, x, theta, lambda, u_hl, Kh, z1, z2, z3, z4, &
                        hl, w_dir(met, ti), .false., time_eval_conc) 
  end if

end function time_eval_conc


! -------------------------------------------------------------------------------------------------------------------------------
subroutine prepare_exceed(c, t_exceed, c_exceed, met, alfa, num_lim, poc_rozpt_t, nphi)
! z pole koncentraci c (prispevek od jednoho zdroje) vyhodnoti t_exceed dle rce 3.61

 implicit none
     
 real(dp), intent(in   ) :: c(nphi, poc_rozpt_t)                 ! pole koncentraci pro 11 rozpt. trid
 real(dp), intent(inout) :: t_exceed(nphi, poc_rozpt_t, num_lim, 2) 
 real(dp), intent(in   ) :: c_exceed(num_lim)                    ! limity pro prekroceni
 real(dp), intent(in   ) :: alfa			         ! relativni rocni vyuziti alfa 
 integer,  intent(in   ) :: met, num_lim, poc_rozpt_t, nphi	
 integer		 :: l, r, ai  

 do l = 1, num_lim
     do r = 1, poc_rozpt_t
         do ai = 1, nphi
             if ((t_exceed(ai, r, l, 1) == 0) .and. (c(ai, r) > c_exceed(l))) then
				t_exceed(ai, r, l, 1) = alfa
				t_exceed(ai, r, l, 2) = met
             endif
         enddo
     enddo
 enddo

end subroutine prepare_exceed


! -------------------------------------------------------------------------------------------------------------------------------

subroutine eval_stat(c, c_d, c_max, cd_max, c_ave, th_exceed, td_exceed, results)
 ! spocte statistiky ktere me zajimaji (prumer, maximalky atd.), z ruzic nikoliv z casove rady

 use mod_init

 implicit none

 real(dp), dimension(:,:,:), intent(in   )    :: c 
 real(dp), dimension(:,:,:), intent(in   )    :: c_d
 real(dp), dimension(:,:),   intent(in   )    :: c_max  
 real(dp), dimension(:,:),   intent(in   )    :: cd_max 
 real(dp), intent(in   ) 		      :: c_ave
 real(dp), dimension(:,:,:, :), intent(in   ) :: th_exceed
 real(dp), dimension(:,:,:, :), intent(in   ) :: td_exceed
 real(dp), dimension(:),     intent(  out)    :: results
 integer                                      :: l, class_max, ind
 real(dp)				      :: c_max_abs, w_sp_max, w_dir_max 
	
 ! absolutni hodinove maximum, trida, rychlost, smer
 if (is_max_abs) then 
    call find_max(c, c_max_abs, class_max, w_sp_max, w_dir_max)
    results(locate_string(stats, 'Ch_max')) = c_max_abs
    results(locate_string(stats, 'Ch_class_max')) = class_max
    results(locate_string(stats, 'Ch_w_sp_max')) = w_sp_max
    results(locate_string(stats, 'Ch_w_dir_max')) = w_dir_max
 endif

 ! hodinova maxima pro kazdou z rozptylovych podminek
 if (is_max_11) then
     ind = locate_string(stats, 'Ch_I_1.7')
     results(ind : ind + poc_rozpt_t - 1) = maxval(c_max, 1)
 endif

 ! absolutni denni maximum, trida, rychlost, smer
 if (is_max_abs .and. is_daily) then
     call find_max(c_d, c_max_abs, class_max, w_sp_max, w_dir_max)
     results(locate_string(stats, 'Cd_max')) = c_max_abs
     results(locate_string(stats, 'Cd_class_max')) = class_max
     results(locate_string(stats, 'Cd_w_sp_max')) = w_sp_max
     results(locate_string(stats, 'Cd_w_dir_max')) = w_dir_max
 endif

 ! denni maxima pro kazdou z rozptylovych podminek
 if (is_max_11 .and. is_daily) then
     ind = locate_string(stats, 'Cd_I_1.7')
     results(ind : ind + poc_rozpt_t - 1) = maxval(cd_max, 1)
 endif

 ! prumerne koncentrace
 if (is_average) then
     ind = locate_string(stats, 'ave')
     results(ind) = c_ave
 endif

 ! doby prekroceni 
 if (is_exceed) then
     ind = locate_string(stats, 'Th1')
     do l = 1, num_lim      
         results(ind + l - 1) = eval_t_exceed(th_exceed(:, :, l, :))
     enddo
     if (is_daily) then
        ind = locate_string(stats, 'Td1')
        do l = 1, num_lim      
            results(ind + l - 1) = eval_t_exceed(td_exceed(:, :, l, :)) 
        enddo
     endif
 endif
 
end subroutine eval_stat

! -------------------------------------------------------------------------------------------------------------------------------

function daily(Ch, Pd, dtype, a, b, c)
 ! z hodinovych koncentraci pocita denni, kapitola 4.6.1
 implicit none
 real(dp), dimension(a, b, c), intent(in   )	:: Ch
 real(dp),                     intent(in   )	:: Pd
 character(4),                 intent(in   )	:: dtype
 integer,                      intent(in   )	:: a, b, c        ! rozmery vstupniho a vystupniho pole
 real(dp), dimension(a, b, c)                   :: daily

 select case (dtype) 
     case ('SO2')                 ! rovnice 4.13
         where (Ch <= 388) 
     	     daily = 0.7439 * Ch * Pd / 24
         elsewhere 
             daily = (0.0342 * Ch + 275.5) * Pd / 24
         end where

     case ('PM10')                ! rovnice 4.14
         where (Ch <= 360)
     	     daily = 0.8364 * Ch * Pd / 24
         elsewhere 
             daily = (0.03482 * (log(Ch))**5.1144) * Pd / 24
         end where

     case default
         write(*,*) 'Chyba ve funkci daily. Zadany typ pro vypocet dennich maximalnich koncentraci, neodpovida ocekavanym hodnotam.'
         stop

 end select 

end function daily



! -------------------------------------------------------------------------------------------------------------------------------

subroutine write_output_stat(results)
 ! vypise do vystupniho souboru oputDir/output_stats vsechny statistiky, ty ktere se nepocitaly budou 0

 use mod_init

 implicit none

 real(dp), dimension(:,:), intent(in   )     :: results
 integer, parameter			     :: poc_stats = size(stats) 
 integer                                     :: pocet_ref_bodu, ri, outUnit, i, ind
 logical				     :: output_cols(poc_stats)	
 character(100)                              :: fmt
 character(500)				     :: path_file	
 character(12)				     :: stats_pom(poc_stats), lim
 character(2)				     :: lim_in

 pocet_ref_bodu = size(results, 2)

 output_cols = .false.
 if (is_max_abs) then 
     ind = locate_string(stats, 'Ch_max')
     output_cols(ind : ind + 3) = .true.
     if (is_daily) then
           ind = locate_string(stats, 'Cd_max')
           output_cols(ind : ind + 3) = .true.
     endif
 endif
 if (is_max_11) then 
     ind = locate_string(stats, 'Ch_I_1.7')
     output_cols(ind : ind + poc_rozpt_t - 1) = .true.
     if (is_daily) then
         ind = locate_string(stats, 'Cd_I_1.7')
         output_cols(ind : ind + poc_rozpt_t - 1) = .true.
     endif
 endif
 if (is_exceed) then
     ind = locate_string(stats, 'Th1')
     output_cols(ind : ind + num_lim - 1) = .true.
     if (is_daily) then
         ind = locate_string(stats, 'Td1')
         output_cols(ind : ind + num_lim - 1) = .true.
     endif
 endif
 if (is_average) then
     ind = locate_string(stats, 'ave')
     output_cols(ind) = .true.
 endif
  
 path_file = trim(oputDir) // trim(output_stats)        
 open(unit = newunit(outUnit), file = path_file, action = 'write')

 ! write headline
 fmt = '(*(G0.5, :, "' // trim(delim_out) // '"))'
 write(outUnit, fmt, advance = 'no') 'ID', 'x[m]', 'y[m]', 'z[m]', 'l[m]'
 ! zameni Th1, Td1, Th2 atd. ca Th_lim1 ...
 stats_pom = stats
 do i = 1, num_lim
     write(lim_in, '(I0)') i
     ind = locate_string(stats, 'Th' // lim_in)
     write(lim, '(F0.3)') ch_exceed(i)
     stats_pom(ind) = 'Th_' // trim(adjustl(lim))
     ind = locate_string(stats, 'Td' // lim_in)
     write(lim, '(F0.3)') cd_exceed(i)
     stats_pom(ind) = 'Td_' // trim(adjustl(lim))
 enddo 
 ! zapise zbytek hlavicky
 do i = 1, poc_stats
     if (output_cols(i)) write(outUnit, '(A, G0.5)', advance = 'no') trim(delim_out), trim(stats_pom(i))
 enddo
 write(outUnit,*)

 ! write data
 fmt = '(A20, "' // trim(delim_out) // '", 2(F0.2, "' // trim(delim_out) // '"), F0.2, "' & 
                 // trim(delim_out) // '", F0.2, "' // trim(delim_out) // '", *(ES11.5, :, "' // trim(delim_out) // '"))'
 do ri = 1, pocet_ref_bodu
     write(outUnit, fmt) & 
           ref_body(ri)%id, ref_body(ri)%x, ref_body(ri)%y, ref_body(ri)%z, ref_body(ri)%l, pack(results(:, ri), output_cols)
 enddo
 close(outUnit)

end subroutine write_output_stat


! -------------------------------------------------------------------------------------------------------------------------------

subroutine write_output_time(oputDir, output_time, ref_ids, date, results)
 ! vypise do vystupniho souboru oputDir/output_time maximalni hodinove koncentrace

 use mod_init, only: delim_out

 implicit none

 character(*),               intent(in   ) :: oputDir, output_time
 character(*), dimension(:), intent(in   ) :: ref_ids
 character(*), dimension(:), intent(in   ) :: date
 real(dp), dimension(:,:),   intent(in   ) :: results
 integer                                   :: ri, ti, pocet_output_bodu, ntime, outUnit
 character(100)				   :: path_file	
 
 pocet_output_bodu = size(ref_ids)
 ntime = size(date)

 path_file = trim(oputDir) // trim(output_time) 
 open(unit = newunit(outUnit), file = path_file, action = 'write')

 ! write headline
 write(outUnit, '(A3)', advance = 'no')  'cas'
 do ri = 1, pocet_output_bodu
     write(outUnit, '(A, G0.5)', advance = 'no') trim(delim_out), trim(ref_ids(ri))
 enddo
 write(outUnit,*)

 ! write data
 do ti = 1, ntime
    write(outUnit, '(A13, *("' // trim(delim_out) // '", ES11.5))') date(ti), results(ti, :)
 enddo
 close(outUnit)

end subroutine write_output_time


! -------------------------------------------------------------------------------------------------------------------------------
subroutine write_time_average(oputDir, output_time, c_r)
 ! do souboru oputdir/avg_output_time vypise prumer pro kazdy ref. bod

 use mod_init, only: ref_body, delim_out

 implicit none

 character(*),               intent(in   ) :: oputDir, output_time
 real(dp), dimension(:),     intent(in   ) :: c_r
 character(150)				   :: fmt
 character(500)				   :: path_file
 integer				   :: outUnit, ri, pocet_ref_bodu

 pocet_ref_bodu = size(c_r, 1)
 path_file = trim(oputDir) // 'avg_' // trim(output_time) 
 open(unit = newunit(outUnit), file = path_file, action = 'write')

 ! write headline
 fmt = '(*(G0.5, :, "' // trim(delim_out) // '"))'
 write(outUnit, fmt) 'ID', 'x[m]', 'y[m]', 'z[m]', 'l[m]', 'ave'

 ! write data
 fmt = '(A20, "' // trim(delim_out) // '", 2(F0.2, "' // trim(delim_out) // '"), F0.2, "' & 
                 // trim(delim_out) // '", F0.2, "' // trim(delim_out) // '", ES11.5)'
 do ri = 1, pocet_ref_bodu
    write(outUnit, fmt) ref_body(ri)%id, ref_body(ri)%x, ref_body(ri)%y, ref_body(ri)%z, ref_body(ri)%l, c_r(ri)
 enddo
 close(outUnit)

end subroutine write_time_average


! -------------------------------------------------------------------------------------------------------------------------------
subroutine write_time_maximum(oputDir, output_time, c_m)
 ! do souboru oputdir/max_output_time vypise maximum pres vsechny casove okamziky pro kazdy ref. bod

 use mod_init, only: ref_body, delim_out

 implicit none

 character(*),               intent(in   ) :: oputDir, output_time
 real(dp), dimension(:),     intent(in   ) :: c_m
 character(150)				   :: fmt
 character(500)				   :: path_file
 integer				   :: outUnit, ri, pocet_ref_bodu

 pocet_ref_bodu = size(c_m, 1)
 path_file = trim(oputDir) // 'max_' // trim(output_time) 
 open(unit = newunit(outUnit), file = path_file, action = 'write')

 ! write headline
 fmt = '(*(G0.5, :, "' // trim(delim_out) // '"))'
 write(outUnit, fmt) 'ID', 'x[m]', 'y[m]', 'z[m]', 'l[m]', 'max'

 ! write data
 fmt = '(A20, "' // trim(delim_out) // '", 2(F0.2, "' // trim(delim_out) // '"), F0.2, "' & 
                 // trim(delim_out) // '", F0.2, "' // trim(delim_out) // '", ES11.5)'
 do ri = 1, pocet_ref_bodu
    write(outUnit, fmt) ref_body(ri)%id, ref_body(ri)%x, ref_body(ri)%y, ref_body(ri)%z, ref_body(ri)%l, c_m(ri)
 enddo
 close(outUnit)

end subroutine write_time_maximum

! -------------------------------------------------------------------------------------------------------------------------------

subroutine find_max(c, c_max_abs, class_max, w_sp_max, w_dir_max)
  ! Najde absolutni maximum - hodnotu a za jakych podminek nastava.

  use mod_init, only: phi_step

  implicit none

  real(dp), dimension(:,:,:), intent(in   )	:: c
  real(dp), intent(  out)                       :: c_max_abs, w_sp_max, w_dir_max
  integer                                       :: class_max
  integer                                       :: c_max_loc(3)
  
  c_max_abs = maxval(c)  
  c_max_loc = maxloc(c)

  w_dir_max = c_max_loc(1) * phi_step - 1 	! smer vetru, pri kterem nastala maximalni koncentrace c_max_abs
  w_sp_max = u_range(c_max_loc(2))		! rychlost vetru, pri ktere nastala maximalni koncentrace c_max_abs
  class_max = c_max_loc(3)			! stabilitni trida, pri ktere nastala maximalni koncentrace c_max_abs
    
end subroutine find_max



! -------------------------------------------------------------------------------------------------------------------------------
!> Z pole maximálních koncentrací a růžice spočte příspěvek k ročnímu průměru
!> ------------------------- 
!> vstup: pole koncentrací a větrná růžice pro všechny směry větru a všechny rozptylové podmínky, alfa 
!> výstup: průměrná koncentrace dle větrné růžice (rovnice 3.60), kapitola 3.3.2

real(dp) function eval_average(c, ruzice, alfa)

 implicit none

 real(dp), dimension(:,:), intent(in   ) :: c, ruzice 
 real(dp), intent(in   )		 :: alfa
 integer		  		 :: j, ai, poc_rozpt_t, nphi

 nphi = size(c, dim = 1)
 poc_rozpt_t = size(c, dim = 2)
 if ((nphi /= size(ruzice, dim = 1)) .or. (poc_rozpt_t /= size(ruzice, dim = 2))) then
    write(*,*) "error subroutine eval_average: rozdilne dimense vstupnich poli"
    stop
 endif

 eval_average = 0

 do j = 1, poc_rozpt_t
    do ai = 1, nphi
        eval_average = eval_average + ruzice(ai, j) * c(ai, j)
    enddo
 enddo

 eval_average = eval_average * alfa

end function eval_average




! ----------------------------------------------------------------------------------------------------------

real(dp) function eval_t_exceed(t_exceed)
 ! pocita doby prekroceni dle kap. 3.3.3. Vstupem je t (rovnice 3.61), vystup doba prekroceni dle rovnice 3.62.

 use mod_init, only: ruzice, poc_rozpt_t, nphi

 implicit none

 real(dp), intent(in   )	:: t_exceed(nphi, poc_rozpt_t, 2)
 integer			:: j, ai, met

 met = 1
 eval_t_exceed = 0
 do j = 1, poc_rozpt_t
    do ai = 1, nphi
        eval_t_exceed = eval_t_exceed + ruzice(ai, j, met) * t_exceed(ai, j, 1) 
    enddo
 enddo
 eval_t_exceed = eval_t_exceed * 8760

end function eval_t_exceed


! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function fac_NO(is_no, k_p, xL, u_hl)
 ! kapitola 4.5 metodiky. Dle rovnice 4.12a pocita multiplikativni faktor u c_NO

 implicit none

 logical, intent(in   )  :: is_no	! [TRUE/FALSE] zda pocitame s konverzi NO NO2
 real(dp), intent(in   ) :: k_p		! [1/s] koeficient prirustku NO2
 real(dp), intent(in   ) :: xL		! [m] vzdalenost referencniho bodu od zdroje ve smeru vetru
 real(dp), intent(in   ) :: u_hl        ! [m/s] rychlost vetru v efektivni vysce korigovane na tvar terenu

 if (is_no) then
     fac_NO = (1 - dexp(-k_p * xL / u_hl)) * 0.9
 else
     fac_NO = 0
 endif

end function fac_NO

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function u_h(h, u, u_height, p)
    ! spocita rychlost ve vysce h ze zadane rychlosti u ve vysce u_height
    ! Metodika SYMOS vztah (3.34)
    implicit none
    
    real(dp), intent(in   ) :: h        ! [m]   vyska, v niz se ma rychlost spocitat
    real(dp), intent(in   ) :: u        ! [m/s] rychlost ve vysce u_height
    real(dp), intent(in   ) :: u_height ! [m] 
    real(dp), intent(in   ) :: p        ! [-]   exponent mocninneho profilu vetru

    if (h <= 10) then
        u_h = u * (10 / u_height)**p
    else if (h < 200) then
        u_h = u * (h / u_height)**p
    else
        u_h = u * (200 / u_height)**p
    endif

end function u_h

! -------------------------------------------------------------------------------------------------------------------------------
!> pocita faktor vzrustu vlecky v pripade vice blizkych zdroju (shluku)
!> ----------
!> Metodika SYMOS kapitola 3.2.3.2

subroutine sfaktor(ind_s, H, dh, E_Ni)
    
    use mod_init
    implicit none
    
    real(dp), intent(in   ) :: dh	!< prevyseni vlecky zdroje, 
    real(dp), intent(in   ) :: H	!< vyska kominu zdroje
    integer,  intent(in   ) :: ind_s 	!< index shluku
    real(dp) :: H_bar, P_Ni, vzd, cond, numer
    integer :: i, j, ns
    real(dp), intent(  out) :: E_Ni	!< faktor vzrustu
    real(dp), dimension(:), allocatable :: zH, zQ, zX, zY
    
    ns = count(zdroje%shluk == ind_s)  ! pocet zdroju ve shluku

    allocate(zH(ns))
    allocate(zQ(ns))
    allocate(zY(ns))
    allocate(zX(ns))  
  
    zH = pack(zdroje%H, zdroje%shluk == ind_s)
    zQ = pack(zdroje%Q, zdroje%shluk == ind_s)
    zX = pack(zdroje%x, zdroje%shluk == ind_s)
    zY = pack(zdroje%y, zdroje%shluk == ind_s)

    H_bar = sum(zH*zQ)/sum(zQ) ! vazeny prumer vysek kominu ve shluku
    cond = 0
    do i = 1, ns
        do j = i + 1, ns
            vzd = sqrt( (zX(i) - zX(j))**2 + (zY(i) - zY(j))**2)
            if (cond < vzd) cond = vzd
        enddo
    enddo

    numer = cond
    if (ind_s > 1000) cond = cond / (ns - 1)  

    if ((cond <= 1.5 * H_bar) .and. (0.5 * H_bar <= H) .and. (1.5 * H_bar >= H)) then
        P_Ni = (6/sqrt(ns*1._dp)) * (numer/dh)**(3/2._dp)
        E_Ni = ((ns+P_Ni)/(1+P_Ni))**(1/3._dp)
    else
        E_Ni = 1
    end if
  
    if (allocated(zQ)) deallocate(zQ)
    if (allocated(zH)) deallocate(zH)
    if (allocated(zY)) deallocate(zY)
    if (allocated(zX)) deallocate(zX)

end subroutine sfaktor

! -------------------------------------------------------------------------------------------------------------------------------

subroutine hef(source, ci, u_in, u_height, x, zmax, h, hl)
    ! spocte efektivni vysku zdroje (zahrne efekt shluku a/nebo chladicich vezi)
    ! vraci jak efektivni vysku bez opravy na teren tak s opravou

    use mod_init, only: zdroj, num_te, num_hum, t_t, r_t, grt, frt, Vch1, Tch1, is_shluky

    implicit none
    
    type(zdroj), intent(in   ) :: source           ! zdroj
    integer,     intent(in   ) :: ci               ! trida stability
    real(dp),    intent(in   ) :: u_in             ! vstupni rychlost vetru 
    real(dp),    intent(in   ) :: u_height         ! vyska ve ktere se zadava vitr
    real(dp),    intent(in   ) :: x                ! vzdalenost zdroje a ref. bodu
    real(dp),    intent(in   ) :: zmax	           ! maximalni vyska terenu nad upatim komina mezi zdrojem a ref. bodem
    real(dp),    intent(  out) :: h, hl            ! h efektivni vyska bez korekce na teren, hl s korekci
    real(dp)	               :: U 	           ! rychlost vetru ve vysce H (koruna komina) 
    real(dp)     	       :: dh               ! prevyseni vlecky 
    real(dp)     	       :: dh_rt            ! prevyseni vlecky pro dane r a t
    real(dp)		       :: Vs		   ! objem spalin za teploty ts 
    real(dp)		       :: E_Ni, u_10
    integer		       :: r, t, ui

    U = u_h(source%H, u_in, u_height, u_p(ci))     ! rychlost vetru ve vysce H (koruna komina)   
    if (.not.source%chl_v) then                    ! normalni komin
        dh = prevyseni(source%w0, source%d, source%beta, source%Q, source%A, source%B, Km(ci), Ks(ci), U, x) 
        if (is_shluky .and. (source%shluk /= 0)) then                ! shluk
            call sfaktor(source%shluk, source%H, dh, E_Ni)
            dh = dh * E_Ni 
        endif
    else                                           ! chladici vez
        dh = 0
        Vs = V_prep(source%Vs, 0._dp, pnorm, source%ts, pnorm)
        u_10 = u_h(10._dp, u_in, u_height, u_p(ci)) ! rychlost vetru ve vysce 10 m
        ui = u2class_ui(u_10)                       ! trida rychlosti vetru v 10 m
        do t = 1, num_te
            do r = 1, num_hum          
                dh_rt = prevyseni_chlad_vez(t_t(t), r_t(r), source%ts, source%H, source%d, Vs, &
                                            frt(r, t), Vch1(r, t, ui), Tch1(r, t, ui), ci, U, x)
                if (is_shluky .and. (source%shluk /= 0)) then        ! shluk
                    call sfaktor(source%shluk, source%H, dh_rt, E_Ni)
                    dh_rt = dh_rt * E_Ni 
                endif
                dh = dh + dh_rt * grt(r, t, ci)                
            enddo
        enddo
    endif 

    if ((source%typ == 2) .and. (source%w0 == 0) .and. (source%d == 0)) dh = source%ts      ! plosny zdroj - zastavba 
    h  = source%H + dh                  ! efektivni vyska
    hl = Hef_terr(h, zmax, eps(ci))     ! korekce na teren

end subroutine hef

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function Hef_terr(h, zmax, eps)
    ! provede korekci efektivni vysky na teren
    ! vystup: efektivni vyska komina s korekci na teren
    ! Metodika SYMOS vztah (3.21)
    implicit none
    
    real(dp), intent(in   ):: h, eps
    real(dp) :: zmax

    Hef_terr = h
    if (zmax > (1 - eps) * h) Hef_terr = zmax + eps * h  

end function Hef_terr

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function get_Kh(Fz, num_Fz_levels, z_z, h, z_r, ci, u_10)
 ! spocte koeficient Kh - zeslabeni vlivu nizkych zdroju na znecisteni v horach, kap 3.2.7., rovnice 3.47

 implicit none

 real(dp), intent(in   )	:: Fz(2, num_Fz_levels)		! kumulativni cetnosti vyskytu inverzi, tab. 3.8
 integer,  intent(in   )	:: num_Fz_levels                ! pocet hodnot ve Fz
 real(dp), intent(in   )        :: z_z				! [m] nadmorska vyska terenu v miste zdroje
 real(dp), intent(in   )        :: h				! [m] efektivni vyska zdroje bez korekce na vliv terenu
 real(dp), intent(in   )        :: z_r				! [m] nadmorska vyska terenu v miste zreferencniho bodu
 integer,  intent(in   )        :: ci				! trida stability
 real(dp), intent(in   )	:: u_10				! [m/s] rychlost vetru ve vysce 10 m nad povrchem zeme

 if (z_r > (z_z + h)) then
     get_Kh = 1 - (Fz1(Fz, num_Fz_levels, z_z + h, ci, u_10) - Fz1(Fz, num_Fz_levels, z_r, ci, u_10))
 else
     get_Kh = 1
 endif

end function get_Kh

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function Fz1(Fz, num_Fz_levels, z, ci, u_10)
 ! spocte funkci F'(z), rovnice 3.48

 implicit none

 real(dp), intent(in   )	:: Fz(2, num_Fz_levels)		! kumulativni cetnosti vyskytu inverzi, tab. 3.8
 integer,  intent(in   )	:: num_Fz_levels                ! pocet hodnot ve F(z)
 real(dp), intent(in   )        :: z				! [m] vyska ve ktere chci funkci F'(z)
 integer,  intent(in   )        :: ci				! trida stability
 real(dp), intent(in   )	:: u_10				! [m/s] rychlost vetru ve vysce 10 m nad povrchem zeme
 integer			:: Fz_ind1, Fz_ind2         
 real(dp)			:: Fz_val, d
 
 if (z <= Fz(1, 1)) then
     Fz_val = Fz(2, 1) 
 else if (z >= Fz(1, num_Fz_levels)) then
     Fz_val = Fz(2, num_Fz_levels)
 else
     Fz_ind1 = minloc(abs(Fz(1, :) - z), 1)             ! najdi v poli F(z) hodnotu nad a pod z
     Fz_ind2 = Fz_ind1 + 1
     if ((Fz(1, Fz_ind1) - z) > 0) then
         Fz_ind2 = Fz_ind1
         Fz_ind1 = Fz_ind2 - 1
     endif

     d = Fz(1, Fz_ind2) - Fz(1, Fz_ind1)
     Fz_val = (d - z + Fz(1, Fz_ind1)) * Fz(2, Fz_ind1) + (d - Fz(1, Fz_ind2) + z) * Fz(2, Fz_ind2)        ! linearni interpolace
     Fz_val = Fz_val / d
 endif

 select case (ci) 
     case (1, 2)
         Fz1 = 2.247 * Fz_val

     case (3)
         if (u_10 <= 2.5) then
             Fz1 = 1.170 * Fz_val
         else if (u_10 < 7.5) then
             Fz1 = 1.170 * Fz_val * (1 - (u_10 - 2.5) / 5)
         else
             Fz1 = 0
         endif

     case (4, 5)
         Fz1 = 0

     case default
         write(*,*) 'Chyba ve funkci Fz1. Trida stability ',ci ,' neodpovida ocekavanym hodnotam.'
         stop
 end select

end function Fz1


! -------------------------------------------------------------------------------------------------------------------------------
real(dp) function get_lambda(dir, h, u_height, d, is_rotation)
 ! pocita lambda dle rovnice 3.55.

 implicit none

 real(dp), intent(in   ) :: dir, h, u_height, d
 logical, intent(in   )  :: is_rotation
 real(dp)		 :: d1

 if (is_rotation) then
    d1 = d - (h - u_height) / 25          ! delta', rce 3.52
 else
    d1 = d
 endif
 get_lambda = mod(abs(dir - d1), 360._dp)

end function get_lambda


! -------------------------------------------------------------------------------------------------------------------------------

subroutine get_tridy_st(tep_gr, tridy_st)
    ! urci tridy stability na zaklade casove rady vertikalniho teplotniho gradientu
    ! Metodika SYMOS, tab. 2.6  
    ! tohle je napsane trochu neintuitivne, a to proto ze tabulka je "nesymetricka" pro tridu IV kde je neostra nerovnost
    implicit none
    
    real(dp),  dimension(:), intent(in   ) :: tep_gr	! [°C/100m]	vertikalni teplotni gradient
    integer, dimension(:), intent(  out)   :: tridy_st	! [-]		trida stability
    integer :: i, t
    
    tridy_st = n_stab_cl - 1           
    do t = 1, ntime		! cyklus pres vsechny casove kroky
    	do i = 1, n_stab_cl - 1
            if (tep_gr(t) < stab_cl_UE(i)) then
                tridy_st(t) = i
                exit
            end if           
        end do	
        if (tep_gr(t) > stab_cl_UE(n_stab_cl - 1)) tridy_st(t) = n_stab_cl			 
    end do
  

end subroutine get_tridy_st

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function V_prep(V1, T1, p1, T2, p2)
    ! vstup: objem plynu V1 za teploty T1 a tlaku p1
    ! vystup: objem plynu V2 za teploty T2 a tlaku p2
    implicit none
    
    real(dp), intent(in   ) :: V1 !< [m3/s] objem za teploty T1 a tlaku p1
    real(dp), intent(in   ) :: T1 !< [°C]   teplota
    real(dp), intent(in   ) :: p1 !< [Pa]   tlak
    real(dp), intent(in   ) :: T2 !< [°C]   teplota
    real(dp), intent(in   ) :: p2 !< [Pa]   tlakp

    V_prep = V1 * (T0K + T2)/(T0K + T1) * p1/p2

end function V_prep


! -------------------------------------------------------------------------------------------------------------------------------

subroutine u_range_gen(is_max_abs, u_range, ind)
    ! generuje rozmezí rychlostí větru pro výpočet maximálních krátkodobých imisních koncentraci
    ! Metodika SYMOS kap. (3.3.1)
    implicit none
    
    integer :: i, class
    integer :: m, n, o
    logical, intent(in) :: is_max_abs
    real(dp), dimension(:), allocatable, intent(  out) :: u_range
    integer, dimension(n_stab_cl), intent(  out) :: ind		! [-] velikost vystupniho pole
    real(dp), dimension(:), allocatable :: urange1
    real(dp), dimension(:), allocatable :: urange2
    real(dp), dimension(:), allocatable :: urange3
     
    if (is_max_abs) then
        m = 16; n = 20; o = 16
        allocate(urange1(m)); allocate(urange2(n)); allocate(urange3(o))        
        urange1 = [ ((1.4 + i*0.1), i = 1, m) ]
        urange2 = [ ((3.0 + i*0.2), i = 1, n) ]
        urange3 = [ ((7.0 + i*0.5), i = 1, o) ]
    else
        m = 1; n = 1; o = 1
        allocate(urange1(m)); allocate(urange2(n)); allocate(urange3(o))
        urange1 = class_w_sp(1)
        urange2 = class_w_sp(2)
        urange3 = class_w_sp(3)
    endif

    allocate(u_range(m+n+o))
    
    u_range(1:m) = urange1
    u_range(m+1:m+n) = urange2
    u_range(m+n+1:) = urange3
    
    do class = 1, n_stab_cl
    
      if (class == 1) then
	ind(class) = maxloc(u_range, 1, mask = u_range.le.2)
      else if (class == 2 .or. class == 5) then
	ind(class) = maxloc(u_range, 1, mask = u_range.le.5)
      else
	ind(class) = maxloc(u_range, 1, mask = u_range.le.15)
      endif
    
    end do

    deallocate(urange1); deallocate(urange2); deallocate(urange3)
      
end subroutine u_range_gen


! -------------------------------------------------------------------------------------------------------------------------------

integer function find(array, value)
    ! najde 1. pozici "value" v poli "array", pokud nenajde vrati -1
    implicit none
    real(dp), dimension(:), intent(in   ) :: array
    real(dp), intent(in   ) :: value
    real(dp), parameter	:: eps = 0.00001
    integer :: i, n
    
    n = size(array)
    do i = 1, n
      if (abs(array(i) - value) < eps) then
	find = i
        return
      endif
    end do
    find = -1
    
end function find

! -------------------------------------------------------------------------------------------------------------------------------
!> Spočte prevýšení osy vlečky nad korunou komína ve vzdálenosti x od zdroje.
!> -----------------------
!> Metodika SYMOS, kap. 3.2.3.1
!> \f[ \Delta h = \left((1-\beta)\frac{1.5w_0 d }{u_H} + \beta\frac{K_s A Q^B }{u_H}\right)\left(\frac{x}{K_m \sqrt{Q}}\right) \mbox{ pro } x < K_m \sqrt{Q}\f]
!> \f[ \Delta h = \left((1-\beta)\frac{1.5w_0 d }{u_H} + \beta\frac{K_s A Q^B }{u_H}\right)  \mbox{ pro }   x > K_m \sqrt{Q}\f]

real(dp) function prevyseni(w0, d, beta, Q, A, B, Km, Ks, u, x)

    implicit none
    
    real(dp), intent(in   ) :: w0  !< [m/s] je výstupní rychlost exhalací
    real(dp), intent(in   ) :: d   !< [m]   vnitřní průměr koruny komína, resp. výduchu
    real(dp), intent(in   ) :: Q   !< [MW]  tepelná vydatnost
    real(dp), intent(in   ) :: u   !< [m/s] rychlost větru ve výšce koruny komína, resp. výduchu
    real(dp), intent(in   ) :: X   !< [m]   vzdalenost ref. bodu od zdroje ve smeru proudeni
                                 !       pokud bude zaporne, vrati funkce maximalni vznos
    !
    ! tyto parametry zavisejici pouze na parametrech zdroje pripadne stabilite zadavam pro urychleni vypoctu,
    ! aby se nemusely pro kazdy referencni bod vyhledavat:
    real(dp), intent(in   ) :: beta !< [-]   koeficiet z rovnice (3.22)
    real(dp), intent(in   ) :: A, B !< [-]   parametry A a B pro vypovet prevyseni (tab. 3.2)
    real(dp), intent(in   ) :: Ks   !< [-]   Konstanta pro vypocet efektivni vysky (tab. 3.1)
    real(dp), intent(in   ) :: Km   !< [-]     --- || ---

    real(dp) :: dh  !> prevyseni vlecky
    real(dp) :: KmQ !> Km*sqrt(Q) 

    KmQ = Km*SQRT(Q)
    ! takto je to v metodicke prirucce, ale dynamicky vznos primo nad kominem pak je nulovy
     dh = (1-beta)*1.5*w0*d/u + beta*Ks*A*Q**B/u
     if (X < KmQ  .and. X > 0D0) dh = dh * (x/KmQ)**(2./3.)

    ! takto je to ve vyzkumne zprave GA/3224-93 za rok 1994 (dynamicky vznos se projevi okamzite):
  !  dh = beta*Ks*A*Q**B/u
  !  if (X < KmQ .and. X > 0D0) dh = dh * (x/KmQ)**(2./3.)
  !  dh = dh + (1-beta)*1.5*w0*d/u

    prevyseni = dh
end function prevyseni

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function prevyseni_chlad_vez(te, r, Ts, H, Dv, Vs, frt, Vch1, Tch1, ci, uH, x)
 ! spočte prevýšení vlečky z chladicich vezi tepelnych elektraren, pro dane te, r (delta h'jrt z rovnice 5.32)
 ! metodika kap. 5.2
 implicit none

 real(dp), intent(in   )  :: te               !< [°C] teplota okolniho vzduchu
 real(dp), intent(in   )  :: r		      !< [%] relativni vlhkost vzduchu
 real(dp), intent(in   )  :: Ts               !< [°C] teplota zavedenych spalin
 real(dp), intent(in   )  :: H                !< [m] vyska chladici veze
 real(dp), intent(in   )  :: Dv               !< [m] prumer chladici veze v korune
 real(dp), intent(in   )  :: Vs               !< [m3/s] objemovy tok zavedencyh spalin
 real(dp), intent(in   )  :: frt              !< prumerne relativni mnozstvi zkondenzovane pary pro dane te a r
 real(dp), intent(in   )  :: Vch1             !< [m3/s] objemovy tok vzduchu bez zavedenych spalin pro dane te a r
 real(dp), intent(in   )  :: Tch1             !< [°C] teplota vzduchu odchazejici z veze bez zavedenych spalin pro dane te a r
 integer,  intent(in   )  :: ci               !< trida stability
 real(dp), intent(in   )  :: uH               !< [m/s] rychlost vetru ve vysce H (vyska chladici veze)
 real(dp), intent(in   )  :: x                !< [m] vzdalenost ref. bodu od veze 

 real(dp) 		  :: TeK              ! [K] teplota vzduchu v okoli chladici veze (te + 273.15)
 real(dp) 		  :: TsK              ! [K] teplota zavedenych spalin (Ts + 273.15) 
 real(dp)                 :: xh1, xh2, sg, F, Tch1K
 
 TeK = te + T0K
 TsK = Ts + T0K 
 Tch1K = Tch1 + T0K
 F = F_vztlak(TeK, r, TsK, Dv, Vs, frt, Vch1, Tch1K)

 select case (ci)                   
     case (1, 2, 3)
         sg = g * (gamma_d - gamma(ci)) / TeK
         xh1 = 2.4 * uH / sqrt(sg)
         if (x < xh1) then                                  ! rovnice 5.12
             prevyseni_chlad_vez = 1.6 * F**(1/3._dp) * x**(2/3._dp) / uH
         else
             prevyseni_chlad_vez = 2.9 * (F / (uH * sg))**(1/3._dp)
         endif
     case (4, 5)   
         xh2 = 2.16 * F**(2/5._dp) * H**(3/5._dp)
         if (x < xh2) then          			     ! rovnice 5.15
             prevyseni_chlad_vez = 1.6 * F**(1/3._dp) * (3 * x)**(2/3._dp) / uH      
         else
             prevyseni_chlad_vez = 5.56 * F**(3/5._dp) * H**(2/5._dp) / uH 
         endif
     case default
         write(*,*) 'Chyba ve funkci prevyseni_chlad_vez. Trida stability ',ci,' neodpovida ocekavanym hodnotam.'
         stop
 end select

end function prevyseni_chlad_vez

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function F_vztlak(Te, r, Ts, Dv, Vs, frt, Vch1, Tch1)
 ! funkce pro vypocet vztlakoveho toku vzduchu opoustejiciho chladici vez (rce 5.17)
 implicit none

 real(dp), intent(in   )  :: Te         !< [K] teplota okolniho vzduchu
 real(dp), intent(in   )  :: r		!< [%] relativni vlhkost vzduchu
 real(dp), intent(in   )  :: Ts         !< [K] teplota zavedenych spalin
 real(dp), intent(in   )  :: Dv         !< [m] prumer chladici veze v korune
 real(dp), intent(in   )  :: Vs         !< [m3/s] objemovy tok zavedenych spalin
 real(dp), intent(in   )  :: frt        !< prumerne relativni mnozstvi zkondenzovane pary
 real(dp), intent(in   )  :: Vch1       !< [m3/s] objemovy tok vzduchu bez zavedenych spalin
 real(dp), intent(in   )  :: Tch1       !< [K] teplota vzduchu odchazejici z veze bez zavedenych spalin
 real(dp)		  :: w, Tch, qch, qe

 w = get_w(Vs + Vch1, Dv)
 Tch = get_tch(Ts, Vs, Vch1, Tch1)
 qe = get_qe(r, Te)
 qch = get_qe(100._dp, Tch)

 F_vztlak = w * g * Dv**2 / 4
 F_vztlak = F_vztlak * (1 - Te / Tch + (qch - qe) * (0.61 + frt * Lv / (cp * Tch)))

end function F_vztlak

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function get_ET(T)
 ! tlak nasycenych vodnich par pri teplote T (rce 5.19)
 implicit none

 real(dp), intent(in   )  :: T       ! [K] teplota

 get_ET = Er * exp(-(1 / T - 1 / T0K) * Lv / Rv)

end function get_ET

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function get_w(Vch, Dv)
 ! funkce pro vypocet vystupni rychlosti z chladici veze w (rce 5.21)
 implicit none

 real(dp), intent(in   )  :: Vch       ! [m3/s] objemovy tok vzduchu opoustejiciho chladici vez (vcetne zavedenych spalin)
 real(dp), intent(in   )  :: Dv        ! [m] prumer chladici veze v korune

 get_w = 4 * Vch / (pi * Dv**2)

end function get_w

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function get_qe(r, T)
 ! smesovaci pomer (rce 5.18)
 implicit none

 real(dp), intent(in   )  :: r		! [%] relativni vlhkost vzduchu
 real(dp), intent(in   )  :: T          ! [K] teplota 

 get_qe = r * 0.622 * get_ET(T) / (100 * pnorm)

end function get_qe

! -------------------------------------------------------------------------------------------------------------------------------
!> Teplota vzduchu odchazejiciho z chladici veze 
!> ------------
!> Metodika SYMOS (vztah 5.24)

real(dp) function get_Tch(Ts, Vs, Vch1, Tch1)
 
 implicit none

 real(dp), intent(in   )  :: Ts         !< [K] teplota zavedenych spalin
 real(dp), intent(in   )  :: Vs         !< [m3/s] objemovy tok zavedenych spalin
 real(dp), intent(in   )  :: Vch1       !< [m3/s] objemovy tok vzduchu bez zavedenych spalin
 real(dp), intent(in   )  :: Tch1       !< [K] teplota vzduchu odchazejici z veze bez zavedenych spalin
 real(dp)		  :: Rs, Rch1, Ms, Mch1

 Rs = get_R(Ts)
 Ms = get_M(Vs, Rs, Ts)
 Rch1 = get_R(Tch1)
 Mch1 = get_M(Vch1, Rch1, Tch1)

 get_Tch = (Ms * Ts + Mch1 * Tch1) / (Ms + Mch1)

end function get_Tch

! -------------------------------------------------------------------------------------------------------------------------------
!> Spocte plynovou konstantu spalin
!> ------------
!> Metodika SYMOS vztahy (5.26 + 5.29)

real(dp) function get_R(T)

 implicit none

 real(dp), intent(in   )  :: T		

 get_R = Rd * (1 + 0.61 * get_s(T))

end function get_R

! -------------------------------------------------------------------------------------------------------------------------------
!> Merna vlhkost nasycenych vodnich par 
!> -------------
!> Metodika SYMOS vztah (5.30)


real(dp) function get_s(T)
 
 implicit none

 real(dp), intent(in   )  :: T		
 real(dp)		  :: ET

 ET = get_ET(T)
 get_s = 0.622 * ET / (pnorm - 0.378 * ET)

end function get_s

! -------------------------------------------------------------------------------------------------------------------------------
!> Hmotnost spalin/vzduchu za jednotku casu 
!> -------------
!> Metodika SYMOS vztah (5.25 + 5.28)

real(dp) function get_M(V, R, T)
 
 implicit none

 real(dp), intent(in   )  :: V, R, T		

 get_M = pnorm * V / (R * T)

end function get_M

! -------------------------------------------------------------------------------------------------------------------------------
!> Spocte horizontalni a vertikalni rozptylove parametry
!> -------------
!> Metodika SYMOS, kap. 3.2.5.1

subroutine get_sgm(X, ay, by, az, bz, sgmy0, sgmz0, sgmy, sgmz)

    implicit none
    
    real(dp), intent(in   ) :: X            !> [m]   vzdalenost ref. bodu od zdroje ve smeru proudeni
    real(dp), intent(in   ) :: ay,by,az,bz  !> [-]   cas prumerovani (1 nebo 8 hodin)
    real(dp), intent(in   ) :: sgmy0, sgmz0 !> [m]   pocatecni hodnoty rozptylovych koeficientu
    real(dp), intent(  out) :: sgmy, sgmz   !> [m]   rozptylove koeficienty
  
    sgmy = sgmy0 + ay*X**by ! (3.37)
    sgmz = sgmz0 + az*X**bz ! (3.37)
end subroutine get_sgm


! -------------------------------------------------------------------------------------------------------------------------------
!> Spocte pocatecni hodnoty rozptylovych parametru
!> --------
!> Metodika SYMOS, kap. 3.2.5.2 + 3.2.5.3

subroutine get_sgm0(izdroj, phi, az, bz, sgmy0, sgmz0)
    
    use mod_init
    
    implicit none 
    
    type(zdroj), intent(in   ) :: izdroj
    real(dp), intent(in   ) 	:: az	!> soucinnovy rozptylovy parametr
    real(dp), intent(in   ) 	:: bz 	!> mocninny rozptylovy parametr
    real(dp), intent(in   ) 	:: phi		!> smer vetru
    real(dp)                     :: psi
    real(dp)			:: zeta, sinz, cosz, sfpi
    real(dp), intent(  out) 	:: sgmy0, sgmz0 ! [m]   pocatecni hodnoty rozptylovych koeficientu

    select case (izdroj%typ)
      case (1) ! bodovy zdroj
	sgmy0 = 0
	sgmz0 = 0
	
      case (2) ! plosny zdroj
	sgmy0 = izdroj%y0 / sqrt(2 * pi)
	sgmz0 = az * (izdroj%y0 / 2) ** bz
	
      case (3) ! liniovy zdroj	
	psi = get_delta_val(izdroj%x1, izdroj%y1, izdroj%x2, izdroj%y2) 
	
	zeta = abs(phi - psi) 
	
	if ((zeta >= 0) .and. (zeta < 90)) then
	  zeta = zeta
	else if ((zeta >= 90) .and. (zeta < 180)) then
	  zeta = 180 - zeta
	else if ((zeta >= 0) .and. (zeta < 270)) then
	  zeta = zeta - 180
	else 
	  zeta = 360 - zeta
	end if
	
	sinz = sin(zeta * pi / 180)
	cosz = cos(zeta * pi / 180)
	sfpi = sqrt(pi/2)	
	
	sgmy0 = (izdroj%y0*sinz + izdroj%x0*cosz)/ sqrt(2*pi)
	sgmz0 = (izdroj%z0 + sfpi*az*(min(izdroj%x0/sinz, izdroj%y0/cosz)/2)**bz)/sfpi
	
      case default
          write(*,*)'_get_sgm0: nedovolena hodnota typu zdroje'
          write(*,*)'              parametr "typ_zdroje" muze nabyvat hodnot 1, 2, 3'
          write(*,*)'              zadana hodnota "class": ', izdroj%typ
          stop
     end select 
end subroutine get_sgm0


! -------------------------------------------------------------------------------------------------------------------------------
! Spocte koeficient odstranovani na zaklade tridy latky
! -------------
! Metodika SYMOS tab. (3.7)

real(dp) function get_coef_dep(class)

    implicit none
    
    integer, intent(in   ) :: class  ! [-]   trida latky
    
    if ((class > size(k_u_all)) .or. (class < 1)) then
          write(*,*)'_get_coef_dep: Nedovolena hodnota tridy znecistujici latky (parametr trida_lat)'
          write(*,*)'              Hodnota tridy muze nabyvat hodnot 1 az max. ', max_k_u
          write(*,*)'              Zadana hodnota tridy: ',class
          stop
    else
        get_coef_dep = k_u_all(class)
    endif
  
end function get_coef_dep

! -------------------------------------------------------------------------------------------------------------------------------

subroutine get_sgm_par(time)

    ! do proměnných ay, by, az, bz přiřadí odpovídající hodnoty rozptylových parametrů    
    implicit none
    
    integer, intent(in  ) :: time  	  ! [min]   	cas prumerovani kratkodobych koncentraci v min
    
    ! ay i ay_time ... jsou globalni promenne

    select case (time)
        case (60)
          ay = ay_1h
          by = by_1h
          az = az_1h
          bz = bz_1h
        case (480)
          ay = ay_8h
          by = by_8h
          az = az_8h
          bz = bz_8h
        case (30)
          ay = ay_30m
          by = by_30m
          az = az_30m
          bz = bz_30m
        case default
          write(*,*)'_get_sgm_par: Nedovolena hodnota prumerovaciho intervalu'
          write(*,*)'              Parametr "time" muze nabyvat hodnot 30, 60, 480'
          write(*,*)'              Zadana hodnota "time": ',time
          stop
    end select

end subroutine get_sgm_par



! -------------------------------------------------------------------------------------------------------------------------------
! Zahrnuti inverznich situaci do vypoctu
! -------------
! Metodika O. Vlcek (2014)

subroutine eval_inv(ti, ri, izdroj, c)

  use mod_init
  
  implicit none
  
  integer, intent(in   ) 	:: ti     !< vstupujici index trid stability
  integer, intent(in   ) 	:: ri     !< vstupujici index referencniho bodu
  type(zdroj), intent(in   ) 	:: izdroj !< vstupujici index zdroje
  real(dp), intent(  out) 	:: c      !< vystupni koncentrace pri vypoctu s inverzeni
  type(zdroj)			:: izdroj_corr
  integer			:: met
  real(dp)			:: theta_i, zmax_i
  real(dp)			:: zr, h, dh0, hl, z1, z2, z3, z4, u_hl, lambda, lr, Kh, L, dz
  logical			:: inv  !indikuje zda do hlavni rovnice zahrnout horni odraz od inverze
     
  met = izdroj%met
  dz = ref_body(ri)%z - izdroj%z
  L = zi(met, ti) - izdroj%z
  dh0 = izdroj%A*izdroj%Q**izdroj%B            ! pro urceni, zda vlecka prorazi inverzi
  call evaluate_par(izdroj, tridy_st(met, ti), ri, w_sp(met, ti), u_height, L, h, hl, z1, z2, z3, z4, Kh, u_hl)
  Kh = 1

  if ((h < L) .or. ((h > L) .and. ((izdroj%H + dh0/2) <= L))) then ! vlecka zustane pod inverzi
    if (dz + ref_body(ri)%l > L) then                       ! referencni bod nad inverzi
      c = 0
    else                                                   ! referencni bod pod inverzi
      if (zmax > L) then                                   ! kopec vycnivajici nad inverzi
	c = 0 
      else                                                 ! kopec pod inverzi    
         if (h > L) then                                   ! vlecka se zarazi o inverzi
            h = L
            hl = Hef_terr(h, zmax, eps(tridy_st(met, ti)))       
            u_hl = u_h(hl, w_sp(met, ti), u_height, u_p(tridy_st(met, ti)))                
            call get_vert_coord(ref_body(ri)%z, izdroj%z, ref_body(ri)%l, hl, L, free_atmo, z1, z2, z3, z4)        
         endif                                                                          
         if (hl > L) then                                  ! kontrola zda po korekci na teren nejsem nad inverzi
            hl = L          
            u_hl = u_h(hl, w_sp(met, ti), u_height, u_p(tridy_st(met, ti)))                
            call get_vert_coord(ref_body(ri)%z, izdroj%z, ref_body(ri)%l, hl, L, free_atmo, z1, z2, z3, z4)
         endif
         lambda = get_lambda(w_dir(met, ti), h, u_height, delta, wind_rotation)	
         inv = .true.   
	 call evaluate_gauss(tridy_st(met, ti), izdroj, x, theta, lambda, u_hl, Kh, z1, z2, z3, z4, &
                             hl, w_dir(met, ti), inv, c)	
      end if      
    end if
    
  else                                                      ! vlecka je nad inverzi 
      if (dz + ref_body(ri)%l < L) then                     ! referencni bod pod inverzi
         c = 0
      else                                                  ! referencni bod nad inverzi	  
         izdroj_corr = izdroj			            ! fiktivni zdroj nad inverzi
         izdroj_corr%z = zi(met, ti)                        ! nove nadmorske vysky pro fiktivni teren 
         if (ref_body(ri)%z < zi(met, ti)) then
            zr = zi(met, ti)
            lr = ref_body(ri)%l - (zi(met, ti) - ref_body(ri)%z)
         else 
            zr = ref_body(ri)%z
            lr = ref_body(ri)%l
         end if
         call get_theta_val(izdroj_corr%z, zr, izdroj_corr%x, izdroj_corr%y, ref_body(ri)%x, ref_body(ri)%y, &
                            zi(met, ti), theta_i, zmax_i) 
   	 if (izdroj%H > L) then                             ! komin je nad inverzi           
            izdroj_corr%H = izdroj%H - L 
         else                                               ! komin je pod inverzi
            izdroj_corr%H = hl - L
            izdroj_corr%ts = 0
            izdroj_corr%w0 = 0
            izdroj_corr%beta = 0
         endif
         ! nad inverzi neuvazuji staceni vetru ani zmenu rychlosti vetru s vyskou
         call hef(izdroj_corr, tridy_st_i(met, ti), w_sp_i(met, ti), izdroj_corr%H, x, zmax_i, h, hl)                    
         call get_vert_coord(zr, izdroj_corr%z, lr, hl, L, .TRUE., z1, z2, z3, z4)                 
         lambda = get_lambda(w_dir_i(met, ti), h, u_height, delta, .FALSE.)	 
         inv = .false.       
         call evaluate_gauss(tridy_st_i(met, ti), izdroj_corr, x, theta_i, lambda, w_sp_i(met, ti), Kh, &
                             z1, z2, z3, z4, hl, w_dir_i(met, ti), inv, c)
      end if    
  end if  

end subroutine eval_inv



! -------------------------------------------------------------------------------------------------------------------------------

subroutine get_vert_coord(zr, zz, l, h, Li, free_atmo, z1, z2, z3, z4)
    ! stanovi vertikalni promenne z', z'', z''' 
    ! Metodika SYMOS kap. 3.2.1.2.    
    
    implicit none
    
    real(dp)		    :: z, zl
    real(dp), intent(in   ) :: zr    !< [m]	nadmorska vyska v miste referencniho bodu
    real(dp), intent(in   ) :: zz    !< [m]	nadmorska vyska terenu v miste zdroje
    real(dp), intent(in   ) :: l     !< [m]	vyska referencniho bodu nad urovni terenu, resp. vyska budovy
    real(dp), intent(in   ) :: h     !< [m]	efektivni vyska zdroje
    real(dp), intent(in   ) :: Li    !< [m]	vyska inverze (pokud se inverze nepocita, je to libovolne cislo a z4 se nepouzije)
    logical,  intent(in   ) :: free_atmo   !<   zda pocitat ve volne amtosfere
    real(dp), intent(  out) :: z1	   !< [m]	korigovana vertikalni souradnice referencniho bodu v clenu pro primy rozptyl
    real(dp), intent(  out) :: z2	   !< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz v dolnim odhadu
    real(dp), intent(  out) :: z3	   !< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz v hornim odhadu
    real(dp), intent(  out) :: z4	   !< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz od inverze

    z = zr - zz
    zl = z + l

    if ((zl <= h) .or. (free_atmo)) then
      z1 = z + l 
      z2 = abs(z) + l
      z3 = z - l
      z4 = z + l - 2 * Li
    else
      z1 = h
      z2 = abs(z) + h - z
      z3 = 2*z - h
      z4 = h - 2 * Li
    end if
    
end subroutine get_vert_coord

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function gfaktor(sgmy, sgmz, U, Vsn, Y, ku, X, Kh)
  
    implicit none
    
    real(dp), intent(in   ) :: X     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru proudeni
    real(dp), intent(in   ) :: U     		!< [m/s] rychlost vetru v efektivni vysce zdroje
    real(dp), intent(in   ) :: Vsn   		!< [Nm3/s] objemovy tok spalin za normalnich podminek (0°C, 101325Pa)
    real(dp), intent(in   ) :: Y     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru kolmem na smer vetru
    real(dp), intent(in   ) :: ku    		!< [1/s] koeficient odstranovani, zahrnujici suchou a mokrou depozici a chemicke transformace
    real(dp), intent(in   ) :: Kh    		!< [-]	koeficient zeslabení vlivu nízkých zdrojů na referenční body ve větších nadmořských výškách
    real(dp), intent(in   ) :: sgmy, sgmz 	!< [m]  rozptylove koeficienty. Již v sobě zahrnují počáteční hodnoty rozptylových parametrů liniových a plošných zdrojů.
    
    gfaktor = 1.D6 / (2*Pi*sgmy*sgmz*U + Vsn) * exp(-Y**2/(2*sgmy**2))* exp(-ku*X/U)* Kh
    
end function gfaktor

! -------------------------------------------------------------------------------------------------------------------------------

real(dp) function inv_exp(X, U, Vsn, Y, ku, Kh, z4, hl, sgmy, sgmz)

    implicit none
    
    real(dp), intent(in   ) :: X     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru proudeni
    real(dp), intent(in   ) :: U     		!< [m/s] rychlost vetru v efektivni vysce zdroje
    real(dp), intent(in   ) :: Vsn   		!< [Nm3/s] objemovy tok spalin za normalnich podminek (0°C, 101325Pa)
    real(dp), intent(in   ) :: Y     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru kolmem na smer vetru
    real(dp), intent(in   ) :: ku    		!< [1/s] koeficient odstranovani, zahrnujici suchou a mokrou depozici a chemicke transformace
    real(dp), intent(in   ) :: Kh    		!< [-]	koeficient zeslabení vlivu nízkých zdrojů na referenční body ve větších nadmořských výškách
    real(dp), intent(in   ) :: hl    		!< [m]	efektivni vyska zdroje po korekci na teren
    real(dp), intent(in   ) :: z4	   	!< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz v hornim odhadu
    real(dp), intent(in   ) :: sgmy, sgmz 	!< [m]  rozptylove koeficienty. Již v sobě zahrnují počáteční hodnoty rozptylových parametrů liniových a plošných zdrojů.
    
    inv_exp =  gfaktor(sgmy, sgmz, U, Vsn, Y, ku, X, Kh) * exp(-(z4+hl)**2/(2*sgmz**2))     

end function

! -------------------------------------------------------------------------------------------------------------------------------
!> obecná rovnice pro výpočet imisních koncentrací plynné látky
!> ------------------
!> Metodika SYMOS vztah (3.1)
!konc_norm(xL, u_hl, izdroj%Vs, yL, k_u, Kh, z1, z2, z3, theta, sgmy, sgmz)
real(dp) function konc_norm(X, U, Vsn, Y, ku, Kh, z1, z2, z3, hl, theta, sgmy, sgmz)
  
    implicit none
    
    real(dp), intent(in   ) :: X     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru proudeni
    real(dp), intent(in   ) :: U     		!< [m/s] rychlost vetru v efektivni vysce zdroje
    real(dp), intent(in   ) :: Vsn   		!< [Nm3/s] objemovy tok spalin za normalnich podminek (0°C, 101325Pa)
    real(dp), intent(in   ) :: Y     		!< [m]	vzdalenost ref. bodu od zdroje ve smeru kolmem na smer vetru
    real(dp), intent(in   ) :: ku    		!< [1/s] koeficient odstranovani, zahrnujici suchou a mokrou depozici a chemicke transformace
    real(dp), intent(in   ) :: Kh    		!< [-]	koeficient zeslabení vlivu nízkých zdrojů na referenční body ve větších nadmořských výškách
    real(dp), intent(in   ) :: hl    		!< [m]	efektivni vyska zdroje po korekci na teren
    real(dp), intent(in   ) :: theta 		!< [-]	koeficient pro zvlneny teren
    real(dp), intent(in   ) :: z1	   	!< [m]	korigovana vertikalni souradnice referencniho bodu v clenu pro primy rozptyl
    real(dp), intent(in   ) :: z2	   	!< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz v dolnim odhadu
    real(dp), intent(in   ) :: z3	   	!< [m]	korigovana vertikalni souradnice referencniho bodu v clenu popisujicim odraz v hornim odhadu
    real(dp), intent(in   ) :: sgmy, sgmz 	!< [m]  rozptylove koeficienty. Již v sobě zahrnují počáteční hodnoty rozptylových parametrů liniových a plošných zdrojů.
    
    konc_norm =  gfaktor(sgmy, sgmz, U, Vsn, Y, ku, X, Kh) &
              * (             exp(-(z1-hl)**2/(2*sgmz**2)) &
                  + (1-theta)*exp(-(z2+hl)**2/(2*sgmz**2)) &
                  +    theta *exp(-(z3-hl)**2/(2*sgmz**2)) &
                )
!    write(*,*) gfaktor(sgmy, sgmz, U, Vsn, Y, ku, X, Kh), (             exp(-(z1-hl)**2/(2*sgmz**2)) &
!                  + (1-theta)*exp(-(z2+hl)**2/(2*sgmz**2)) &
!                  +    theta *exp(-(z3-hl)**2/(2*sgmz**2)) &
!                )

end function konc_norm


! -------------------------------------------------------------------------------------------------------------------------------
!> Subroutina zajistujici vypocet theta, delta, zmax. 

subroutine eval_theta_delta(xr, yr, zr, xs, ys, zs, theta, delta, zmax, x)
 
 implicit none

 real(dp), intent(in   ) :: xr, yr, zr     !< souradnice a nadmorska vyska referencniho bodu
 real(dp), intent(in   ) :: xs, ys, zs     !< souradnice a nadmorska vyska zdroje
 real(dp), intent(in   ) :: x              !< vzdalenost zdroje a referencniho bodu
 real(dp), intent(  out) :: theta, delta, zmax
 
 delta = get_delta_val(xr, yr, xs, ys)
 call get_theta_val(zs, zr, xs, ys, xr, yr, 0._dp, theta, zmax)

end subroutine eval_theta_delta


! -------------------------------------------------------------------------------------------------------------------------------
!> Funkce pro vypocet x (vzdalenost zdroje od referencniho bodu). 
 real(dp) function eval_x(xr, yr, xs, ys)
 
  implicit none

  real(dp), intent(in   ) :: xr, yr    !< souradnice referencniho bodu
  real(dp), intent(in   ) :: xs, ys    !< souradnice zdroje
  real(dp)	          :: xd, yd
 
  xd = xs - xr
  yd = ys - yr
  eval_x  = sqrt(xd**2 + yd**2)

 end function eval_x


! -------------------------------------------------------------------------------------------------------------------------------
!> Vypocet delta
!> --------------
!> Metodika SYMOS vztah (3.50)

  real(dp) function get_delta_val(xr, yr, xs, ys)

    implicit none
    
    real(dp), intent(in   ) :: xr, yr, xs, ys
    real(dp)	            :: xd, yd
 
    xd = xs - xr
    yd = ys - yr 
    
    if (xd == 0) then        ! POROVNAVANI FLOATING POINT PROMENNE
      get_delta_val = 90 - 90 * signum(yd)
    else if (yd == 0) then
      get_delta_val = 180 - 90 * signum(xd)
    else
      get_delta_val = datan(xd / yd) * 180 / pi + 90 * (2 - signum(xd) * (1 + signum(yd)))
    end if

  end function get_delta_val

  
! -------------------------------------------------------------------------------------------------------------------------------
!> Spocte theta a zmax

  subroutine get_theta_val(zs, zr, xs, ys, xr, yr, z0_level, theta, z_max)
  
    implicit none
    
    real(dp), intent(in   ) 	:: zs, zr, xs, ys, xr, yr !<
    real(dp), intent(in   ) 	:: z0_level               !< výška pro přepočet výškopisu (hodnoty <z0_level jsou nahrazeny z0_level)
    real(dp), intent(  out) 	:: theta                  !<
    real(dp), intent(  out) 	:: z_max                  !<
    real(dp)			:: integral               
    
    call integrace(zs, zr, xs, ys, xr, yr, z0_level, integral, z_max)
    if (zr <= zs) then
      theta = 0
    else
      theta = max(0., integral)
    end if	

  end subroutine get_theta_val
  

! -------------------------------------------------------------------------------------------------------------------------------
!> Vypocet koeficientu vlivu terenu
!> ---------------
!> Metodika SYMOS vztah (3.19)

  subroutine integrace(zs, zr, xs, ys, xr, yr, z0_level, integral, z_max)

    use mod_init
    
    implicit none
    
    real(dp), intent(in   )  			:: zs, zr, xs, ys
    real(dp), intent(in   ) 	:: z0_level             !< výška pro přepočet výškopisu (hodnoty <z0_level jsou nahrazeny z0_level)
    real(dp), intent(  out) :: integral
    real(dp), intent(  out) :: z_max
    real(dp), dimension(:), allocatable :: xc, yc, zc	! pocitane souradnice a pole terenu
  !  real(dp), dimension(3) :: xa, ya			! body vstupujici do interpolace 
  !  real(dp), dimension(3,3) :: za			! vyska terenu vstupujici do interpolace    	
    real(dp) 					:: s_int, z1, z2, h, rd, hx, hy, xr, yr  ! zc_int_err
    integer 					:: i, n, xc_ind, yc_ind

    s_int = 0.	! soucet pres teren
    rd = sqrt((xr - xs)**2 + (yr - ys)**2) !vzdalenost zdroje a referencniho bodu    

    h = int_step(int_err)	! vypocetni krok
    n = rd/h	! pocet pocitanych bodu
    hx = (xr - xs)/n	! prumet horizontalniho kroku na osu x
    hy = (yr - ys)/n	! prumet horizontalniho kroku na osu y
    
    allocate(zc(0:n+1))
    allocate(xc(0:n+1))
    allocate(yc(0:n+1))
    
    xc(0) = xs
    yc(0) = ys
    zc(0) = zs
     
    do i = 1, n

       xc(i) = xc(i-1)+hx
       yc(i) = yc(i-1)+hy
       xc_ind = terrain_ind(xc(i), xllcorner, cellsize)
       yc_ind = terrain_ind(yc(i), yllcorner, cellsize)        
       zc(i) = z(xc_ind, yc_ind)   
       if (zc(i) < z0_level) zc(i) = z0_level          ! fiktivni teren (pro ucely vypoctu nad inverzi)

      ! pripravena interpolace
      !(pro urychleni vypoctu je vhodne vybirat pouze body v blizkosti interpolovavaneho bodu)
!       if ((xc_ind /= 0) .and. (yc_ind /= 0)) then
! 	xa = x(xc_ind-1:xc_ind+1)
! 	ya = y(yc_ind-1:yc_ind+1)
! 	za = z(xc_ind-1:xc_ind+1,yc_ind-1:yc_ind+1) 
!   
!       else if (xc_ind == 0) then
! 	xa = x(xc_ind:xc_ind+2)
! 	ya = y(yc_ind-1:yc_ind+1)
! 	za = z(xc_ind:xc_ind+2,yc_ind-1:yc_ind+1) 
!       
!       else if (yc_ind == 0) then
! 	xa = x(xc_ind-1:xc_ind+1)
! 	ya = y(yc_ind:yc_ind+2)
! 	za = z(xc_ind-1:xc_ind+1,yc_ind:yc_ind+2) 	
!       
!       else
! 	xa = x(xc_ind:xc_ind+2)
! 	ya = y(yc_ind:yc_ind+2)
! 	za = z(xc_ind:xc_ind+2,yc_ind:yc_ind+2)
!       end if      
!       call polin2(xa, ya, za, xc(i), yc(i), zc(i), zc_int_err)

      if (zc(i) > zs) then
	z1 = zc(i) - zs
      else
	z1 = 0.
      end if
      
      if (zc(i) > zr) then
	z2 = zc(i) - zr
      else
	z2 = 0.
      end if
     
      s_int = s_int + h * (z1 - 2 * z2) 
 
    end do
  
    ! posledni bod n+1 (referencni bod)
     xc(n+1) = xr
     yc(n+1) = yr
     zc(n+1) = zr
     h = sqrt((xc(n)-xc(n+1))**2+(yc(n)-yc(n+1))**2)
     if (zr > zs) then
	s_int = s_int + h * (zr - zs) / 2
     else if (zr < zs) then
        s_int = s_int + h * (zr - zs)
     end if      

    integral = s_int / (rd*(zr-zs))
    z_max = maxval(zc) - zs

    if (allocated(zc)) deallocate(zc)
    if (allocated(xc)) deallocate(xc)
    if (allocated(yc)) deallocate(yc)
    
    
  end subroutine integrace


! -------------------------------------------------------------------------------------------------------------------------------
!> Funkce výpočet velikosti integračního kroku
!> ---------------
!> Vychází se z chyby lichoběžníkového pravidla e <= (M x^3) / (12 n^2) ... n počet kroků, x koncových bodů, M horní odhad druhé derivace.
! zatim bere zadanou hodnotu

real(dp) function int_step(x)

 implicit none
 real(dp), intent(in   ) :: x

 int_step = x
end function int_step


! -------------------------------------------------------------------------------------------------------------------------------
!> Funkce pro přepočet rychlosti větru na třídu rychlosti vetru, tabulka 2.5

integer function u2class_ui(u)
  
  implicit none
  real(dp), intent(in   ) :: u
  integer		  :: ui

  if (u < 0) then
     write(*,*) 'Chyba ve funkci u2class_ui.'
     stop
  endif

  do ui = 1, max_class_w_sp - 1
     if (u <= class_w_sp_UE(ui)) then
        u2class_ui = ui 
        return
     endif
  enddo
  u2class_ui = max_class_w_sp

end function u2class_ui



! -------------------------------------------------------------------------------------------------------------------------------
!> Funkce signum
!> ----------------
!> Pomocna funkce

real(dp) function signum(x)
  
  implicit none
  real(dp), intent(in   ) :: x

  signum = sign(1._dp, x) 
  if(x == 0._dp) signum = 0  

end function signum


! -------------------------------------------------------------------------------------------------------------------------------

integer function newunit(unit)
! This is a simple function to search for an available unit.
! LUN_MIN and LUN_MAX define the range of possible LUNs to check.
! The unit value is returned by the function, and also by the optional
! argument. This allows the function to be used directly in an open
! statement, and optionally save the result in a local variable.
! If no units are available, -1 is returned.
  integer, intent(out), optional :: unit
! local
  integer, parameter :: LUN_MIN = 10, LUN_MAX = 1000
  logical :: opened
  integer :: lun
! begin
  newunit = -1
  do lun=LUN_MIN,LUN_MAX
    inquire(unit=lun, opened=opened)
    if (.not. opened) then
      newunit=lun
      EXIT
    end if
  end do
  if (present(unit)) unit=newunit
end function newunit


end module mod_funkce
